<?php
/**
 * Masteriyo Mailchimp integration setting class.
 *
 * @since 2.7.1
 * @package Masteriyo\Addons\MailchimpIntegration
 */

namespace Masteriyo\Addons\MailchimpIntegration;

defined( 'ABSPATH' ) || exit;

/**
 * Masteriyo Mailchimp Integration setting class.
 *
 * @class Masteriyo\Addons\MailchimpIntegrationAddon\Setting
 */

class Setting {
	/**
	 * Setting option name.
	 *
	 * @var string
	 */
	private $name = 'masteriyo_mailchimp_integration_setting';

	/**
	 * Setting data.
	 *
	 * @since 2.7.1
	 *
	 * @var array
	 */
	private $data = array(
		'forced_email_subscription'  => false,
		'api_key'                    => '',
		'subscriber_consent_message' => 'I would like to receive the newsletters.',
		'default_group'              => array( 'value' => '' ),
		'default_list'               => array( 'value' => '' ),
	);


	/**
	 * Initialize.
	 *
	 * @since 2.7.1
	 */
	public function init() {
		add_filter( 'masteriyo_rest_pre_insert_setting_object', array( $this, 'save' ), 10, 3 );
		add_filter( 'masteriyo_rest_prepare_setting_object', array( $this, 'append_setting_in_response' ), 10, 3 );
	}

	/**
	 * Append Mailchimp Integration setting to the global settings.
	 *
	 * @since 2.7.1
	 *
	 * @param WP_REST_Response $response The response object.
	 * @param Model            $object   Object data.
	 * @param WP_REST_Request  $request  Request object.
	 * @return WP_REST_Response
	 */
	public function append_setting_in_response( $response, $object, $request ) {
		$data = $response->get_data();
		$data['integrations']['mailchimp_integration'] = $this->get();

		$response->set_data( $data );

		return $response;
	}

	/**
	 * Store Mailchimp integration settings.
	 *
	 * @since 2.7.1
	 *
	 * @param Model         $setting  Object object.
	 * @param WP_REST_Request $request  Request object.
	 * @param bool            $creating If is creating a new object.
	 *
	 * @return Masteriyo\Models\Setting
	 */
	public function save( $setting, $request, $creating ) {
		$setting_in_db = get_option( $this->name, $this->data );
		$request_body  = json_decode( $request->get_body(), true );

		$post_data = isset( $request_body['advance']['mailchimp_integration'] )
		? $request_body['advance']['mailchimp_integration']
		: array();

		$setting_arr = wp_parse_args( $post_data, $setting_in_db );

		update_option( $this->name, $setting_arr );
		return $setting;
	}


	/**
	 * Return setting value.
	 *
	 * @since 2.7.1
	 * @param string $key Setting key.
	 * @param mixed $default Setting default.
	 * @return mixed
	 */
	public function get( $key = null, $default = null ) {
		$setting = get_option( $this->name, array() );
		$setting = wp_parse_args( $setting, $this->data );

		$value = $key ? masteriyo_array_get( $setting, $key, $default ) : $setting;

		return $value;
	}

	/**
	 * Save setting value.
	 *
	 * @since 2.7.1
	 *
	 * @param string $key Setting key.
	 * @param mixed $default Setting default.
	 */
	public function set( $key, $value ) {
		masteriyo_array_set( $this->data, $key, $value );
		update_option( $this->name, $this->data );
	}
}
