<?php
/**
 * GoogleProcessLogin class.
 *
 * @since 2.7.0
 */

namespace Masteriyo\Addons\SocialLogin;

use League\OAuth2\Client\Provider\Google;
use Masteriyo\Addons\SocialLogin\Setting;

class GoogleProcessLogin {

	/**
	 * Initialization.
	 *
	 * @since 2.7.0
	 *
	 * @return void
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'process' ) );
	}

	/**
	 * Process the google login request and  logs in the user to home location.
	 *
	 * @since 2.7.0
	 *
	 */
	public static function process() {
		if ( ! self::is_google_login_path() || ! isset( $_GET['code'] ) ) {//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			return;
		}
		$setting = new Setting();
		$setting->init();
		$google_database_info = $setting->get( 'google' );

		$provider = self::create_google_client( $google_database_info );

		if ( isset( $_GET['code'] ) ) {//phpcs:ignore WordPress.Security.NonceVerification.Recommended
			self::handle_google_response( $provider );
		}
	}

	/**
	 * sees if user is login using google
	 *
	 * @since 2.7.0
	 *
	 * updated
	 * @since 2.17.0
	 */
	private static function is_google_login_path() {
		$structure = get_option( 'permalink_structure' );

		$request_uri  = wp_parse_url( $_SERVER['REQUEST_URI'], PHP_URL_PATH );
		$query_string = wp_parse_url( $_SERVER['REQUEST_URI'], PHP_URL_QUERY );

		if ( 'plain' === $structure || '' === $structure && isset( $query_string ) ) {
			parse_str( $query_string, $query_params );
			return isset( $query_params['redirect_key'] ) && 'masteriyo-google-login' === $query_params['redirect_key'];
		}
		if ( 'plain' !== $structure ) {
			$trimmed_path = trim( $request_uri, "\n\r\t\v\x00\/" );
			return 'masteriyo-google-login' === $trimmed_path;
		}
	}

	/**
	 * create a instance of google client
	 *
	 * @since 2.7.0
	 */
	private static function create_google_client( $google_database_info ) {
		$provider = new Google(
			array(
				'clientId'     => $google_database_info['client_id'],
				'clientSecret' => $google_database_info['secret_key'],
				'redirectUri'  => masteriyo_get_social_login_redirect_url( 'masteriyo-google-login' ),
				'scopes'       => array( 'openid', 'email', 'profile' ),
			)
		);
		return $provider;
	}

	/**
	 * Handle google response on the user data on the basis of existing database info
	 *
	 * @param \League\OAuth2\Client\Provider\Google; $provider The new provider object.
	 * @since 2.7.0
	 *
	 */
	private static function handle_google_response( $provider ) {
		$code         = $_GET['code']; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$role         = sanitize_text_field( $_GET['state'] ); //phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$access_token = $provider->getAccessToken(
			'authorization_code',
			array(
				'code' => $code,
			)
		);

		// Get the access token and refresh token
		if ( ! $access_token ) {
			return 'Error: No access token.';
		}

		if ( $access_token ) {
			$resource_owner = $provider->getResourceOwner( $access_token );
			$email          = $resource_owner->getEmail();
			$name           = strstr( $email, '@', true );

			$firstname     = $resource_owner->getFirstName();
			$lastname      = $resource_owner->getLastName();
			$profile_image = $resource_owner->getAvatar();

			$user = get_user_by( 'email', $email );

			if ( $user ) {
				masteriyo_set_customer_auth_cookie( $user->ID );
				$page = masteriyo_get_setting( 'general.pages.account_page_id' );
				if ( $page ) {
					wp_redirect( get_permalink( $page ) );
					exit;
				} else {
					wp_redirect( home_url( '/' ) );
					exit;
				}
			}

			$user_id = wp_insert_user(
				array(
					'user_email'   => sanitize_email( $email ),
					'user_login'   => sanitize_text_field( $name ),
					'user_pass'    => '',
					'display_name' => sanitize_text_field( $name ),
					'role'         => sanitize_text_field( $role ),
					'first_name'   => sanitize_text_field( $firstname ),
					'last_name'    => sanitize_text_field( $lastname ),
				)
			);

			if ( is_wp_error( $user_id ) ) {
				echo esc_html( $user_id->get_error_message() );
			}
			wp_set_auth_cookie( $user_id, true );
			$page = masteriyo_get_setting( 'general.pages.account_page_id' );
			if ( $page ) {
				wp_redirect( get_permalink( $page ) );
				exit;
			} else {
				wp_redirect( home_url( '/' ) );
				exit;
			}
		}
	}
}
