<?php
/**
 * Masteriyo Social Login integration setting class.
 *
 * @since 2.7.0
 * @package Masteriyo\Addons\SocialLogin
 */

namespace Masteriyo\Addons\SocialLogin;

defined( 'ABSPATH' ) || exit;

/**
 * Masteriyo Social Login setting class.
 *
 * @class Masteriyo\Addons\SocialLogin\Setting.
 *
 * @since 2.7.0
 */

class Setting {
	/**
	 * Setting option name.
	 *
	 * @var string
	 *
	 * @since 2.7.0
	 */
	private $name = 'masteriyo_social_login_setting';

	/**
	 * Setting data.
	 *
	 * @since 2.7.0
	 *
	 * @var array
	 */
	private $data = array(
		'google'   => array(
			'enable'     => false,
			'client_id'  => '',
			'secret_key' => '',
		),
		'facebook' => array(
			'enable'     => false,
			'client_id'  => '',
			'secret_key' => '',
		),
	);


	/**
	 * Initialize.
	 *
	 * @since 2.7.0
	 */
	public function init() {
		add_filter( 'masteriyo_rest_pre_insert_setting_object', array( $this, 'save' ), 10, 3 );
		add_filter( 'masteriyo_rest_prepare_setting_object', array( $this, 'append_setting_in_response' ), 10, 3 );
	}

	/**
	 * Append Social Login setting to the global settings.
	 *
	 * @since 2.7.0
	 *
	 * @param WP_REST_Response $response The response object.
	 * @param Model            $object   Object data.
	 * @param WP_REST_Request  $request  Request object.
	 * @return WP_REST_Response
	 */
	public function append_setting_in_response( $response, $object, $request ) {
		$data                                   = $response->get_data();
		$data['authentication']['social_login'] = $this->get();
		$data['authentication']['social_login']['google']['redirect_url']   = masteriyo_get_social_login_redirect_url( 'masteriyo-google-login' );
		$data['authentication']['social_login']['facebook']['redirect_url'] = masteriyo_get_social_login_redirect_url( 'masteriyo-facebook-login' );

		$response->set_data( $data );

		return $response;
	}

	/**
	 * Store social login settings.
	 *
	 * @since 2.7.0
	 *
	 * @param Model         $setting  Object object.
	 * @param WP_REST_Request $request  Request object.
	 * @param bool            $creating If is creating a new object.
	 *
	 * @return Masteriyo\Models\Setting
	 */
	public function save( $setting, $request, $creating ) {
		$setting_in_db = get_option( $this->name, $this->data );
		$post_data     = masteriyo_array_get( $request, 'authentication.social_login', array() );
		$setting_arr   = wp_parse_args( $post_data, $setting_in_db );

		update_option( $this->name, $setting_arr );

		return $setting;
	}

	/**
	 * Return setting value.
	 *
	 * @since 2.7.0
	 * @param string $key Setting key.
	 * @param mixed $default Setting default.
	 * @return mixed
	 */
	public function get( $key = null, $default = null ) {
		$setting = get_option( $this->name, array() );
		$setting = wp_parse_args( $setting, $this->data );

		$value = $key ? masteriyo_array_get( $setting, $key, $default ) : $setting;

		return $value;
	}

	/**
	 * Save setting value.
	 *
	 * @since 2.7.0
	 *
	 * @param string $key Setting key.
	 * @param mixed $default Setting default.
	 */
	public function set( $key, $value ) {
		masteriyo_array_set( $this->data, $key, $value );
		update_option( $this->name, $this->data );
	}
}
