<?php
/**
 * Social share service provider.
 *
 * @since 2.5.27
 */

namespace Masteriyo\Addons\SocialShare\Providers;

defined( 'ABSPATH' ) || exit;

use Masteriyo\PostType\PostType;
use League\Container\ServiceProvider\AbstractServiceProvider;
use League\Container\ServiceProvider\BootableServiceProviderInterface;
/**
 * Password Strength  service provider.
 *
 * @since 2.5.27
 */
class SocialShareServiceProvider extends AbstractServiceProvider implements BootableServiceProviderInterface {
	/**
	 * The provided array is a way to let the container
	 * know that a service is provided by this service
	 * provider. Every service that is registered via
	 * this service provider must have an alias added
	 * to this array or it will be ignored
	 *
	 * @since 2.5.27
	 *
	 * @var array
	 */
	protected $provides = array();

	/**
	 * This is where the magic happens, within the method you can
	 * access the container and register or retrieve anything
	 * that you need to, but remember, every alias registered
	 * within this method must be declared in the `$provides` array.
	 *
	 * @since 2.5.27
	 */
	public function register() {
	}

	/**
	 * In much the same way, this method has access to the container
	 * itself and can interact with it however you wish, the difference
	 * is that the boot method is invoked as soon as you register
	 * the service provider with the container meaning that everything
	 * in this method is eagerly loaded.
	 *
	 * If you wish to apply inflectors or register further service providers
	 * from this one, it must be from a bootable service provider like
	 * this one, otherwise they will be ignored.
	 *
	 * @since 2.5.27
	 */
	public function boot() {
		add_action( 'wp_head', array( $this, 'add_meta_tags' ) );
	}

	/**
	 * Add meta tags.
	 *
	 * @since 2.5.27
	 */
	public function add_meta_tags() {
		global $post;

		if ( self::is_seo_plugin_activate() || ! is_singular( PostType::COURSE ) ) {
			return;
		}

		$course = masteriyo_get_course( $post );

		if ( ! $course ) {
			return;
		}

		$image_url        = $course->get_featured_image_url();
		$current_page_url = $course->get_permalink();
		$site_name        = get_bloginfo( 'name' );
		$post_title       = $course->get_name();

		$post_description = $course->get_short_description();
		if ( empty( $post_description ) ) {
			$post_description = wp_trim_words( wp_strip_all_tags( $course->get_description() ), 55 );
		}

		ob_start();

		printf( '<meta property="og:title" content="%s" />', esc_attr( $post_title ) );
		printf( '<meta property="og:description" content="%s" />', esc_attr( $post_description ) );
		printf( '<meta property="og:image" content="%s" />', esc_url( $image_url ) );
		printf( '<meta property="og:url" content="%s" />', esc_attr( $current_page_url ) );
		printf( '<meta property="og:site_name" content="%s" />', esc_attr( $site_name ) );
		printf( '<meta property="og:type" content="%s" />', esc_attr( 'article' ) );

		echo ob_get_clean(); // phpcs:ignore
	}

	/**
	 * Return true if social share plugin is active, false otherwise.
	 *
	 * @since 2.5.27
	 *
	 * @return bool
	 */
	public static function is_seo_plugin_activate() {
		$seo_plugins = array(
			'seo-by-rank-math/rank-math.php',
			'squirrly-seo/squirrly.php',
			'wordpress-seo/wp-seo.php',
			'google-sitemap-generator/sitemap.php',
			'all-in-one-seo-pack/all_in_one_seo_pack.php',
			'wp-seopress/seopress.php',
			'autodescription/autodescription.php',
			'link-whisper/link-whisper.php',
			'surferseo/surferseo.php',
		);

		$active = false;

		foreach ( $seo_plugins as $seo_plugin ) {
			if ( is_plugin_active( $seo_plugin ) ) {
				$active = true;
				break;
			}
		}

		return apply_filters( 'masteriyo_is_seo_plugin_active', $active );
	}
}
