<?php
/**
 * Masteriyo social share addon setup.
 *
 * @package Masteriyo\Addons\SocialShare
 *
 * @since 2.5.27
 */
namespace Masteriyo\Addons\SocialShare;

use Masteriyo\Constants;
use Masteriyo\Addons\SocialShare\Setting;
use Masteriyo\Addons\SocialShare\Enums\SocialService;
use Masteriyo\Addons\SocialShare\SocialShares;

defined( 'ABSPATH' ) || exit;

/**
 * Main Masteriyo SocialShare class.
 *
 * @class Masteriyo\Addons\SocialShare
 */
class SocialShareAddon {

	/**
	 * Instance
	 *
	 * @since 2.5.27
	 *
	 * @var \Masteriyo\Addons\SocialShare\SocialShareAddon
	 */
	protected static $instance = null;

	/**
	 * Constructor.
	 *
	 * @since 2.5.27
	 */
	private function __construct() {
	}

	/**
	 * Return the instance.
	 *
	 * @since 2.5.27
	 *
	 * @return \Masteriyo\Addons\SocialShare\SocialShareAddon
	 */
	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Initialize module.
	 *
	 * @since 2.5.27
	 */
	public function init() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 2.5.27
	 */
	public function init_hooks() {
		add_action( 'masteriyo_single_course_sidebar_content', array( $this, 'render_social_shares_in_single_course_page' ), 25 );
		add_action( 'masteriyo_layout_1_single_course_aside_items', array( $this, 'render_social_shares_in_single_course_page' ), 40 );
		add_filter( 'masteriyo_rest_response_setting_data', array( $this, 'append_setting_in_response' ), 10, 4 );
		add_action( 'masteriyo_new_setting', array( $this, 'save_social_share_settings' ), 10, 1 );

		add_action( 'masteriyo_elementor_social_share_widget', array( $this, 'render_social_shares_in_single_course_page' ), 10, 1 );
		add_filter( 'elementor_course_widgets', array( $this, 'append_custom_course_widgets' ), 10 );
	}

	/**
	 * Add social share elementor widget.
	 *
	 * @since 2.12.2
	 *
	 * @param array $widgets
	 * @return array
	 */
	public function append_custom_course_widgets( $widgets ) {
		$widgets[] = new SocialShareMetaWidget();
		return $widgets;
	}

	/**
	 * Render social shares in single course page.
	 *
	 * @since 2.5.27
	 *
	 * @param \Masteriyo\Models\Course $course
	 */
	public function render_social_shares_in_single_course_page( $course ) {
		$title         = Setting::get( 'title' );
		$shape         = Setting::get( 'shape' );
		$scheme        = Setting::get( 'scheme' );
		$size          = Setting::get( 'size' );
		$size          = $size ? $size . 'px' : '30px';
		$social_shares = SocialShares::all();

		require MASTERIYO_SOCIAL_SHARE_TEMPLATES . '/social-share.php';
	}


	/**
	 * Append setting to response.
	 *
	 * @since 2.5.27
	 *
	 * @param array $data Setting data.
	 * @param \Masteriyo\Models\Setting $setting Setting object.
	 * @param string $context What the value is for. Valid values are view and edit.
	 * @param \Masteriyo\RestApi\Controllers\Version1\SettingsController $controller REST settings controller object.
	 *
	 * @return array
	 */
	public function append_setting_in_response( $data, $setting, $context, $controller ) {
		$data['single_course']['social_share'] = Setting::all();

		return $data;
	}

	/**
	 * Save global social share settings.
	 *
	 * @since 2.5.27
	 *
	 * @param \Masteriyo\Models\Setting $setting Setting object.
	 */
	public function save_social_share_settings( $setting ) {
		$request = masteriyo_current_http_request();

		if ( ! masteriyo_is_rest_api_request() ) {
			return;
		}

		if ( ! isset( $request['single_course']['social_share'] ) ) {
			return;
		}

		$settings = masteriyo_array_only( $request['single_course']['social_share'], array_keys( Setting::all() ) );
		$settings = masteriyo_parse_args( $settings, Setting::all() );

		// Sanitization.
		$settings['title']                                       = sanitize_text_field( $settings['title'] );
		$settings['size']                                        = absint( $settings['size'] );
		$settings['shape']                                       = sanitize_text_field( $settings['shape'] );
		$settings['scheme']                                      = sanitize_text_field( $settings['scheme'] );
		$settings['sharing_services'][ SocialService::FACEBOOK ] = masteriyo_string_to_bool( $settings['sharing_services'][ SocialService::FACEBOOK ] );
		$settings['sharing_services'][ SocialService::TWITTER ]  = masteriyo_string_to_bool( $settings['sharing_services'][ SocialService::TWITTER ] );
		$settings['sharing_services'][ SocialService::LINKEDIN ] = masteriyo_string_to_bool( $settings['sharing_services'][ SocialService::LINKEDIN ] );

		// Validation.
		Setting::set_props( $settings );
		Setting::save();
	}
}
