<?php
/**
 * Store global white label options.
 *
 * @since 2.2.9
 * @package \Masteriyo\Addons\WhiteLabel
 */

namespace Masteriyo\Addons\WhiteLabel;

use Masteriyo\Roles;

class GlobalSetting {

	/**
	 * Global option name.
	 *
	 * @since 2.2.9
	 */
	const OPTION_NAME = 'masteriyo_white_label_settings';

	/**
	 * Data.
	 *
	 * @since 2.2.9
	 *
	 * @var array
	 */
	public $data = array(
		'title'                => 'Masteriyo',
		'inactive_icon'        => 0,
		'active_icon'          => 0,
		'logo'                 => 0,
		'student_role_name'    => '',
		'instructor_role_name' => '',
	);

	/**
	 * Initialize global setting.
	 *
	 * @since 2.2.9
	 */
	public function init() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 2.2.9
	 */
	public function init_hooks() {
		add_filter( 'masteriyo_rest_response_setting_data', array( $this, 'append_setting_in_response' ), 10, 4 );
		add_action( 'masteriyo_new_setting', array( $this, 'save_white_label_settings' ), 10, 1 );
	}

	/**
	 * Append setting to response.
	 *
	 * @since 2.2.9
	 *
	 * @param array $data Setting data.
	 * @param Masteriyo\Models\Setting $setting Setting object.
	 * @param string $context What the value is for. Valid values are view and edit.
	 * @param Masteriyo\RestApi\Controllers\Version1\SettingsController $controller REST settings controller object.
	 *
	 * @return array
	 */
	public function append_setting_in_response( $data, $setting, $context, $controller ) {
		$data['advance']['white_label'] = $this->get();

		return $data;
	}

	/**
	 * Save global white label settings.
	 *
	 * @since 2.2.9
	 *
	 * @param \Masteriyo\Models\Setting $setting Setting object.
	 */
	public function save_white_label_settings( $setting ) {
		$request = masteriyo_current_http_request();

		if ( ! masteriyo_is_rest_api_request() ) {
			return;
		}

		if ( ! isset( $request['advance']['white_label'] ) ) {
			return;
		}

		$settings = masteriyo_array_only( $request['advance']['white_label'], array_keys( $this->data ) );
		$settings = wp_parse_args( $settings, $this->get() );

		// Sanitization
		$settings['title']                = sanitize_text_field( $settings['title'] );
		$settings['logo']                 = absint( $settings['logo'] );
		$settings['active_icon']          = absint( $settings['active_icon'] );
		$settings['inactive_icon']        = absint( $settings['inactive_icon'] );
		$settings['student_role_name']    = trim( sanitize_text_field( $settings['student_role_name'] ) );
		$settings['instructor_role_name'] = trim( sanitize_text_field( $settings['instructor_role_name'] ) );

		// Validate the icons and logos.
		$logo_post                 = get_post( $settings['logo'] );
		$settings['logo']          = ( $logo_post && 'attachment' === $logo_post->post_type ) ? $logo_post->ID : 0;
		$active_icon_post          = get_post( $settings['active_icon'] );
		$settings['active_icon']   = ( $active_icon_post && 'attachment' === $active_icon_post->post_type ) ? $active_icon_post->ID : 0;
		$inactive_icon_post        = get_post( $settings['inactive_icon'] );
		$settings['inactive_icon'] = ( $inactive_icon_post && 'attachment' === $inactive_icon_post->post_type ) ? $inactive_icon_post->ID : 0;

		update_option( self::OPTION_NAME, $settings );
		$this->update_roles();
	}

	/**
	 * Update roles with the new changes.
	 *
	 * @since 2.7.1
	 */
	protected function update_roles() {
		foreach ( Roles::get_all() as $role_slug => $role ) {
			remove_role( $role_slug );
			add_role( $role_slug, $role['display_name'], $role['capabilities'] );
		}
	}

	/**
	 * Return global white field value.
	 *
	 * @since 2.2.9
	 *
	 * @param string $key
	 * @return string
	 */
	public function get( $key = null ) {
		$settings   = get_option( self::OPTION_NAME, $this->data );
		$this->data = wp_parse_args( $settings, $this->data );

		if ( null === $key ) {
			return $this->data;
		}

		if ( isset( $this->data[ $key ] ) ) {
			return $this->data[ $key ];
		}

		return null;
	}

	/**
	 * Set global white label field.
	 *
	 * @since 2.2.9
	 *
	 * @param string $key Setting key.
	 * @param mixed $value Setting value.
	 */
	public function set( $key, $value ) {
		if ( isset( $this->data[ $key ] ) ) {
			$this->data[ $key ] = $value;
		}
	}
}
