<?php
/**
 * Zooms Controller class.
 *
 * @since 2.5.19
 *
 * @package Masteriyo\Addons\Zoom\RestApi
 */

namespace Masteriyo\Addons\Zoom\RestApi;

defined( 'ABSPATH' ) || exit;

use Masteriyo\Addons\Zoom\Enums\ZoomMeetingStatus;
use Masteriyo\Addons\Zoom\Enums\ZoomMeetingType;
use Masteriyo\Addons\Zoom\Enums\ZoomRecordingType;
use Masteriyo\Addons\Zoom\Query\ZoomQuery;
use Masteriyo\Enums\CourseProgressStatus;
use Masteriyo\Enums\PostStatus;
use Masteriyo\Enums\SectionChildrenPostType;
use Masteriyo\Helper\Permission;
use Masteriyo\PostType\PostType;
use Masteriyo\Query\CourseProgressQuery;
use Masteriyo\RestApi\Controllers\Version1\PostsController;

/**
 * ZoomController class.
 */
class ZoomController extends PostsController {
	/**
	 * Endpoint namespace.
	 *
	 * @since 2.5.19
	 *
	 * @var string
	 */
	protected $namespace = 'masteriyo/pro/v1';

	/**
	 * Route base.
	 *
	 * @since 2.5.19
	 *
	 * @var string
	 */
	protected $rest_base = 'zooms';

	/**
	 * Post type.
	 *
	 * @since 2.5.19
	 *
	 * @var string
	 */
	protected $post_type = 'mto-zoom';

	/**
	 * Object type.
	 *
	 * @since 2.5.19
	 *
	 * @var string
	 */
	protected $object_type = 'zoom';

	/**
	 * If object is hierarchical.
	 *
	 * @since 2.5.19
	 *
	 * @var bool
	 */
	protected $hierarchical = true;

	/**
	 * Permission class.
	 *
	 * @since 2.5.19
	 *
	 * @var Masteriyo\Helper\Permission;
	 */
	protected $permission = null;

	/**
	 * Constructor.
	 *
	 * @since 2.5.19
	 *
	 * @param Permission $permission
	 */
	public function __construct( Permission $permission = null ) {
		$this->permission = $permission;
	}

	/**
	 * Register routes.
	 *
	 * @since 2.5.19
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/delete',
			array(
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_items' ),
					'permission_callback' => array( $this, 'delete_items_permissions_check' ),
					'args'                => array(
						'ids'   => array(
							'required'    => true,
							'description' => __( 'Meetings IDs.', 'learning-management-system' ),
							'type'        => 'array',
						),
						'force' => array(
							'default'     => false,
							'description' => __( 'Whether to bypass trash and force deletion.', 'learning-management-system' ),
							'type'        => 'boolean',
						),
					),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args'   => array(
					'id' => array(
						'description' => __( 'Unique identifier for the resource.', 'learning-management-system' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param(
							array(
								'default' => 'view',
							)
						),
					),
				),
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::EDITABLE ),
				),
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
					'args'                => array(
						'force' => array(
							'default'     => true,
							'description' => __( 'Whether to bypass trash and force deletion.', 'learning-management-system' ),
							'type'        => 'boolean',
						),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		// @since 2.5.19 Added clone endpoint to zoom REST API.
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)/clone',
			array(
				'args' => array(
					'id' => array(
						'description' => __( 'Unique identifier for the resource.', 'learning-management-system' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'clone_item' ),
					'permission_callback' => array( $this, 'clone_item_permissions_check' ),
				),
			)
		);

		/**
		 * Since 2.6.5
		 */
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/mine',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_user_zooms' ),
					'permission_callback' => 'is_user_logged_in',
					'args'                => $this->get_collection_params(),
				),
			)
		);
	}

	/**
	 * Add new zoom session.
	 *
	 * @since 2.5.19
	 *
	 * @param WP_REST_Request $request Request object.
	 * @param bool            $creating If is creating a new object.
	 *
	 * @return WP_Error|\Masteriyo\Addons\Zoom\Models\Zoom
	 */
	protected function prepare_object_for_database( $request, $creating = false ) {
		$id   = isset( $request['id'] ) ? absint( $request['id'] ) : 0;
		$zoom = masteriyo( 'zoom' );

		if ( 0 !== $id ) {
			$zoom->set_id( $id );
			$zoom_repo = masteriyo( 'zoom.store' );
			$zoom_repo->read( $zoom );
		}

		// Zoom meeting name
		if ( isset( $request['name'] ) ) {
			$zoom->set_name( $request['name'] );
		}

		// Zoom meeting description
		if ( isset( $request['description'] ) ) {
			$zoom->set_description( wp_slash( $request['description'] ) );
		}

		// Zoom meeting duration
		if ( isset( $request['duration'] ) ) {
			$zoom->set_duration( $request['duration'] );
		}

		// Zoom meeting password
		if ( isset( $request['password'] ) ) {
			$zoom->set_password( $request['password'] );
		}

		// Zoom meeting status
		if ( isset( $request['status'] ) ) {
			$zoom->set_status( $request['status'] );
		}

		// Automatically set the menu order if it's not set and the operation is POST.
		if ( ! isset( $request['menu_order'] ) && $creating ) {
			$query = new \WP_Query(
				array(
					'post_type'      => SectionChildrenPostType::all(),
					'post_status'    => PostStatus::all(),
					'posts_per_page' => 1,
					'post_parent'    => $request['parent_id'],
				)
			);

			$zoom->set_menu_order( $query->found_posts );
		}

		// Zoom meeting parent id
		if ( isset( $request['parent_id'] ) ) {
			$zoom->set_parent_id( $request['parent_id'] );
		}

		// Zoom meeting course id
		if ( isset( $request['course_id'] ) ) {
			$zoom->set_course_id( $request['course_id'] );
		}

		// Zoom meeting join before host setting
		if ( isset( $request['join_before_host'] ) ) {
			$zoom->set_join_before_host( $request['join_before_host'] );
		}

		// Zoom meeting show host video setting
		if ( isset( $request['host_video'] ) ) {
			$zoom->set_host_video( $request['host_video'] );
		}

		// Zoom meeting type
		if ( isset( $request['type'] ) ) {
			$zoom->set_type( $request['type'] );
		}

		// Zoom meeting focus mood
		if ( isset( $request['focus_mode'] ) ) {
			$zoom->set_focus_mode( $request['focus_mode'] );
		}

		// Zoom meeting time zone
		if ( isset( $request['time_zone'] ) ) {
			$zoom->set_time_zone( $request['time_zone'] );
		}

		// Zoom meeting start time
		if ( isset( $request['starts_at'] ) ) {
			$zoom->set_starts_at( $request['starts_at'] );
		}

		// Zoom meeting mute upon entry setting
		if ( isset( $request['mute_upon_entry'] ) ) {
			$zoom->set_mute_upon_entry( $request['mute_upon_entry'] );
		}

		// Zoom meeting participant video setting
		if ( isset( $request['participant_video'] ) ) {
			$zoom->set_participant_video( $request['participant_video'] );
		}

		// Zoom meeting id
		if ( isset( $request['meeting_id'] ) ) {
			$zoom->set_meeting_id( $request['meeting_id'] );
		}

		// Zoom meeting start url.
		if ( isset( $request['start_url'] ) ) {
			$zoom->set_start_url( $request['start_url'] );
		}

		// Zoom meeting Join Url
		if ( isset( $request['join_url'] ) ) {
			$zoom->set_join_url( $request['join_url'] );
		}

		// Zoom meeting record setting.
		if ( isset( $request['record'] ) ) {
			$zoom->set_record( $request['record'] );
		}

		// Zoom meeting close registration.
		if ( isset( $request['close_registration'] ) ) {
			$zoom->set_close_registration( $request['close_registration'] );
		}

		// Allow set meta_data.
		if ( isset( $request['meta_data'] ) && is_array( $request['meta_data'] ) ) {
			foreach ( $request['meta_data'] as $meta ) {
				$zoom->update_meta_data( $meta['key'], $meta['value'], isset( $meta['id'] ) ? $meta['id'] : '' );
			}
		}

		/**
		 * Filters an object before it is inserted via the REST API.
		 *
		 * The dynamic portion of the hook name, `$this->post_type`,
		 * refers to the object type slug.
		 *
		 * @since 2.5.19
		 *
		 * @param Masteriyo\Database\Model $zoom Zoom object.
		 * @param WP_REST_Request $request  Request object.
		 * @param bool            $creating If is creating a new object.
		 */
		return apply_filters( "masteriyo_rest_pre_insert_{$this->post_type}_object", $zoom, $request, $creating );
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @since 2.5.19
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		// The zoom session should be order by menu which is the sort order.
		$params['order']['default']   = 'asc';
		$params['orderby']['default'] = 'menu_order';

		// filter meeting according to meeting status
		$params['status'] = array(
			'description'       => __( 'Limit result by status.', 'learning-management-system' ),
			'type'              => 'enum',
			'enum'              => ZoomMeetingStatus::all(),
			'validate_callback' => 'rest_validate_request_arg',
			'sanitize_callback' => 'sanitize_key',
		);

		// filter meeting according to user's enrolled courses ID.
		$params['course'] = array(
			'description'       => __( 'Limit result set to specific course ids.', 'learning-management-system' ),
			'type'              => 'array',
			'items'             => array(
				'type' => 'integer',
			),
			'default'           => array(),
			'sanitize_callback' => 'wp_parse_id_list',
		);

		return $params;
	}

	/**
	 * Check if a given request has access to create an item.
	 *
	 * @since 2.5.19
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function create_item_permissions_check( $request ) {
		if ( is_null( $this->permission ) ) {
			return new \WP_Error(
				'masteriyo_null_permission',
				__( 'Sorry, the permission object for this resource is null.', 'learning-management-system' )
			);
		}

		if ( masteriyo_is_current_user_admin() || masteriyo_is_current_user_manager() ) {
			return true;
		}

		if ( ! $this->permission->rest_check_post_permissions( $this->post_type, 'create' ) ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_create',
				__( 'Sorry, you are not allowed to create resources.', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		$course_id = absint( $request['course_id'] );
		$course    = masteriyo_get_course( $course_id );

		if ( is_null( $course ) ) {
			return new \WP_Error(
				"masteriyo_rest_{$this->post_type}_invalid_id",
				__( 'Invalid course ID', 'learning-management-system' ),
				array(
					'status' => 404,
				)
			);
		}

		if ( $course->get_author_id() !== get_current_user_id() ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_create',
				__( 'Sorry, you are not allowed to create zoom session for others course.', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		return true;
	}

	/**
	 * Checks if a given request has access to get a specific item.
	 *
	 * @since 2.11.0
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 * @return boolean|\WP_Error True if the request has read access for the item, WP_Error object otherwise.
	 */
	public function get_item_permissions_check( $request ) {
		if ( is_null( $this->permission ) ) {
			return new \WP_Error(
				'masteriyo_null_permission',
				__( 'Sorry, the permission object for this resource is null.', 'learning-management-system' )
			);
		}

		$zoom = masteriyo_get_zoom( $request['id'] );

		if ( is_null( $zoom ) ) {
			return new \WP_Error(
				'masteriyo_rest_invalid_zoom_id',
				__( 'Invalid zoom ID.', 'learning-management-system' ),
				array(
					'status' => 400,
				)
			);
		}

		$course = masteriyo_get_course( $zoom->get_course_id() );

		if ( is_null( $course ) ) {
			return new \WP_Error(
				'masteriyo_rest_invalid_course_id',
				__( 'Invalid course ID.', 'learning-management-system' ),
				array(
					'status' => 400,
				)
			);
		}

		if ( ! $this->permission->rest_check_post_permissions( $this->post_type, 'read', $request['id'] ) ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_read',
				__( 'Sorry, you are not allowed to read resources.', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		return true;
	}

	/**
	 * Get object.
	 *
	 * @since 2.5.19
	 *
	 * @param  int|Model|WP_Post $object Object ID or Model or WP_Post object.
	 * @return object Model object or WP_Error object.
	 */
	protected function get_object( $object ) {
		try {
			if ( is_int( $object ) ) {
				$id = $object;
			} else {
				$id = is_a( $object, '\WP_Post' ) ? $object->ID : $object->get_id();
			}
			$zoom = masteriyo( 'zoom' );
			$zoom->set_id( $id );
			$zoom_repo = masteriyo( 'zoom.store' );
			$zoom_repo->read( $zoom );
		} catch ( \Exception $e ) {
			return false;
		}

		return $zoom;
	}

	/**
	 * Prepares the object for the REST response.
	 *
	 * @since  2.5.19
	 *
	 * @param  Masteriyo\Database\Model $object  Model object.
	 * @param  WP_REST_Request $request Request object.
	 *
	 * @return WP_Error|WP_REST_Response Response object on success, or WP_Error object on failure.
	 */
	protected function prepare_object_for_response( $object, $request ) {
		$context  = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data     = $this->get_zoom_data( $object, $context );
		$data     = $this->add_additional_fields_to_object( $data, $request );
		$data     = $this->filter_response_by_context( $data, $context );
		$response = rest_ensure_response( $data );
		$response->add_links( $this->prepare_links( $object, $request ) );

		/**
		 * Filter the data for a response.
		 *
		 * The dynamic portion of the hook name, $this->post_type,
		 * refers to object type being prepared for the response.
		 *
		 * @since 2.5.19
		 *
		 * @param WP_REST_Response $response The response object.
		 * @param Masteriyo\Database\Model $object   Object data.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( "masteriyo_rest_prepare_{$this->object_type}_object", $response, $object, $request );
	}

		/**
	 * Get zoom data.
	 *
	 * @since 2.5.19
	 *
	 * @param Masteriyo\Models\Zoom $zoom zoom instance.
	 * @param string  $context Request context.
	 * Options: 'view' and 'edit'.
	 *
	 * @return array
	 */
	protected function get_zoom_data( $zoom, $context = 'view' ) {
		$section = masteriyo_get_section( $zoom->get_parent_id() );
		$author  = masteriyo_get_user( $zoom->get_author_id( $context ) );
		$author  = is_wp_error( $author ) || is_null( $author ) ? null : array(
			'id'           => $author->get_id(),
			'display_name' => $author->get_display_name(),
			'avatar_url'   => $author->get_avatar_url(),
		);

		$course = masteriyo_get_course( $zoom->get_course_id( $context ) );
		$data   = array(
			'id'                 => $zoom->get_id(),
			'name'               => wp_specialchars_decode( $zoom->get_name( $context ) ),
			'permalink'          => $zoom->get_permalink( $context ),
			'preview_link'       => $zoom->get_preview_link(),
			'menu_order'         => $zoom->get_menu_order( $context ),
			'parent_menu_order'  => $section ? $section->get_menu_order( $context ) : 0,
			'description'        => $zoom->get_description( $context ),
			'host_video'         => $zoom->get_host_video( $context ),
			'parent_id'          => $zoom->get_parent_id( $context ),
			'course_id'          => $zoom->get_course_id( $context ),
			'created_at'         => masteriyo_rest_prepare_date_response( $zoom->get_created_at( $context ) ),
			'modified_at'        => masteriyo_rest_prepare_date_response( $zoom->get_modified_at( $context ) ),
			'status'             => $this->change_status( $zoom ),
			'course_name'        => $course ? $course->get_name() : '',
			'duration'           => $zoom->get_duration( $context ),
			'join_before_host'   => $zoom->get_join_before_host( $context ),
			'focus_mode'         => $zoom->get_focus_mode( $context ),
			'starts_at'          => masteriyo_rest_prepare_date_response( $zoom->get_starts_at( $context ) ),
			'expires_at'         => masteriyo_rest_prepare_date_response( $zoom->get_expires_at( $context ) ),
			'time_zone'          => $zoom->get_time_zone( $context ),
			'mute_upon_entry'    => $zoom->get_mute_upon_entry( $context ),
			'participant_video'  => $zoom->get_participant_video( $zoom, $context ),
			'record'             => $zoom->get_record( $zoom, $context ),
			'password'           => $zoom->get_password( $zoom, $context ),
			'close_registration' => $zoom->get_close_registration( $zoom, $context ),
			'meeting_id'         => $zoom->get_meeting_id( $zoom, $context ),
			'start_url'          => $zoom->get_start_url( $context ),
			'join_url'           => $zoom->get_join_url( $zoom, $context ),
			'author'             => $author,
			'navigation'         => $this->get_navigation_items( $zoom, $context ),
		);

		/**
		 * Filter zoom rest response data.
		 *
		 * @since 2.5.19
		 *
		 * @param array $data zoom data.
		 * @param Masteriyo\Models\Zoom $zoom zoom object.
		 * @param string $context What the value is for. Valid values are view and edit.
		 * @param Masteriyo\RestApi\Controllers\Version1\ZoomController $controller REST zooms controller object.
		 */
		return apply_filters( "masteriyo_rest_response_{$this->object_type}_data", $data, $zoom, $context, $this );
	}

	/**
	 * Change the zoom status based on get_dynamic_status function and set the status in database if status is needed to update.
	 * @param Masteriyo\Models\Zoom $zoom zoom object.
	 * @since 2.7.1
	 */
	private function change_status( $zoom ) {
		$status = $zoom->get_dynamic_status();

		if ( 'upcoming' === $status && 'upcoming' !== $zoom->get_status() ) {
			$zoom->set_status( 'upcoming' );
			$zoom->save();
		}
		if ( 'active' === $status && 'active' !== $zoom->get_status() ) {
			$zoom->set_status( 'active' );
			$zoom->save();
		}
		if ( 'expired' === $status && 'expired' !== $zoom->get_status() ) {
			$zoom->set_status( 'expired' );
			$zoom->save();
		}
		return $status;
	}

	/**
	 * Get the zooms'schema, conforming to JSON Schema.
	 *
	 * @since 2.5.19
	 *
	 * @return array
	*/
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => $this->object_type,
			'type'       => 'object',
			'properties' => array(
				'id'                 => array(
					'description' => __( 'Unique identifier for the resource.', 'learning-management-system' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
					'readonly'    => true,
				),
				'name'               => array(
					'description' => __( 'Meeting Name', 'learning-management-system' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'description'        => array(
					'description' => __( 'Meeting Description.', 'learning-management-system' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'time_zone'          => array(
					'description' => __( 'User TimeZone', 'learning-management-system' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'password'           => array(
					'description' => __( 'Meeting Password', 'learning-management-system' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'duration'           => array(
					'description' => __( 'Set meeting duration.', 'learning-management-system' ),
					'type'        => 'number',
					'context'     => array( 'view', 'edit' ),
				),
				'join_before_host'   => array(
					'description' => __( 'Allow user to join before host.', 'learning-management-system' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'focus_mode'         => array(
					'description' => __( 'Meeting focus mood', 'learning-management-system' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'starts_at'          => array(
					'description' => __( 'Meeting start time', 'learning-management-system' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'mute_upon_entry'    => array(
					'description' => __( 'Mute user while joining the meeting.', 'learning-management-system' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'participant_video'  => array(
					'description' => __( 'View meeting participant video.', 'learning-management-system' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'host_video'         => array(
					'description' => __( 'View meeting host video.', 'learning-management-system' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'record'             => array(
					'description' => __( 'Allow recording session', 'learning-management-system' ),
					'type'        => 'enum',
					'enum'        => ZoomRecordingType::all(),
					'context'     => array( 'view', 'edit' ),
				),
				'type'               => array(
					'description' => __( 'Meeting type (1)instant meeting, (2) scheduled meeting, (3) recurring meeting with no fixed time, (8) recurring meeting with fixed time.', 'learning-management-system' ),
					'type'        => 'enum',
					'enum'        => ZoomMeetingType::all(),
					'context'     => array( 'view', 'edit' ),
				),
				'close_registration' => array(
					'description' => __( 'Close meeting registration after meeting deadline reached.', 'learning-management-system' ),
					'type'        => 'boolean',
					'context'     => array( 'view', 'edit' ),
				),
				'parent_id'          => array(
					'description' => __( 'Zoom parent ID', 'learning-management-system' ),
					'type'        => 'integer',
					'required'    => true,
					'context'     => array( 'view', 'edit' ),
				),
				'course_id'          => array(
					'description' => __( 'Course ID', 'learning-management-system' ),
					'type'        => 'integer',
					'required'    => true,
					'context'     => array( 'view', 'edit' ),
				),
				'course_name'        => array(
					'description' => __( 'Course name', 'learning-management-system' ),
					'type'        => 'string',
					'readonly'    => true,
					'context'     => array( 'view', 'edit' ),
				),
				'menu_order'         => array(
					'description' => __( 'Menu order, used to custom sort zooms.', 'learning-management-system' ),
					'type'        => 'integer',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Process objects collection.
	 *
	 * @since 2.5.19
	 *
	 * @param array $objects Zoom data.
	 * @param array $query_args Query arguments.
	 * @param array $query_results Zoom query result data.
	 *
	 * @return array
	 */
	protected function process_objects_collection( $objects, $query_args, $query_results ) {
		return array(
			'data' => $objects,
			'meta' => array(
				'total'        => $query_results['total'],
				'pages'        => $query_results['pages'],
				'current_page' => $query_args['paged'],
				'per_page'     => $query_args['posts_per_page'],
				'zoomCounts'   => $this->get_zoom_counts(),
			),
		);
	}

	/**
	 * Get zoom count by status.
	 *
	 * @since 2.5.19
	 *
	 * @return array
	 */
	protected function get_zoom_counts() {
		$post_count = $this->get_zoom_count();
		return masteriyo_array_only( $post_count, array_merge( array( 'any' ), ZoomMeetingStatus::all() ) );

	}

	/**
	 * Get zoom sessions count by status.
	 *
	 * @since 2.5.19
	 *
	 * @return array
	 */
	protected function get_zoom_count() {
		$post_count = parent::get_posts_count();

		$post_count['all'] = array_sum( array( $post_count[ ZoomMeetingStatus::PUBLISH ] ) );

		/**
		 * Filters the zoom counts.
		 *
		 * @since 2.5.19
		 *
		 * @param array $post_count Zoom count.
		 * @param \Masteriyo\RestApi\Controllers\Version1\PostsController $controller Posts Controller.
		 */
		return apply_filters( "masteriyo_rest_{$this->object_type}_count", $post_count, $this );
	}

	/**
	 * Prepare objects query.
	 *
	 * @since  2.5.19
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return array
	 */
	protected function prepare_objects_query( $request ) {
		$args = parent::prepare_objects_query( $request );

		// Set zoom status.
		$args['post_status'] = $request['status'] ?? 'all';

		if ( ! empty( $request['status'] ) ) {
			if ( ZoomMeetingStatus::UPCOMING === $request['status'] ) {

				$args['meta_query'] = array(
					'relation' => 'AND',
					array(
						'key'     => '_starts_at',
						'value'   => time(),
						'type'    => 'numeric',
						'compare' => '>',
					),
				);
			}
		}

		if ( ! empty( $request['course'] ) ) {
				$args['meta_query'] = array(
					'relation' => 'AND',
					array(
						'key'     => '_course_id',
						'value'   => $request['course'],
						'compare' => 'IN',
					),
				);
		}

		if ( ! isset( $request['course'] ) && masteriyo_is_current_user_instructor() ) {
			$args['author__in'] = array( get_current_user_id() );
		}

		if ( ! empty( $request['orderby'] ) ) {
			if ( 'meta_value' === $request['orderby'] ) {
				$args['meta_key'] = '_starts_at';
			}
		}

		return $args;
	}

	/**
	 * Get a collection of user's zooms.
	 *
	 * @since 2.6.5
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 * @param string  $context Request context. Default is 'view'.
	 *
	 * @return \WP_Error|\WP_REST_Response
	 */
	public function get_user_zooms( $request, $context = 'view' ) {
		$course_ids = masteriyo_get_all_user_course_ids( get_current_user_id() );

		$request->set_param( 'course', $course_ids );

		return $this->get_items( $request );
	}

	/**
	 * Prepare objects query for batch.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since 2.7.3
	 * @return array
	 */
	protected function prepare_objects_query_for_batch( $request ) {
		$query_args = array(
			'post_status'    => 'any',
			'post_type'      => $this->post_type,
			'post__in'       => wp_parse_id_list( $request['ids'] ),
			'posts_per_page' => -1,
		);

		/**
		 * Filters objects query for batch operation.
		 *
		 * @since 2.7.3
		 *
		 * @param array $query_args Query arguments.
		 * @param WP_REST_Request $request
		 * @param \Masteriyo\RestApi\Controllers\Version1\PostsController $controller
		 */
		return apply_filters( 'masteriyo_rest_objects_query_for_batch', $query_args, $request, $this );
	}
}
