<?php
/**
 * Zoom setting controller class.
 *
 * @since 2.5.19
 *
 * @package Masteriyo\Addons\Zoom\RestApi
 */

namespace Masteriyo\Addons\Zoom\RestApi;

defined( 'ABSPATH' ) || exit;

use Masteriyo\Addons\Zoom\Models\Setting;
use Masteriyo\RestApi\Controllers\Version1\CrudController;
use ThemeGrill\Zoom\Request;
use ThemeGrill\Zoom\ServerToServerAuth;
use WP_Error;

/**
 * ZoomSetting Controller class.
 */
class ZoomSettingController extends CrudController {
	/**
	 * Endpoint namespace.
	 *
	 * @since 2.5.19
	 *
	 * @var string
	 */
	protected $namespace = 'masteriyo/pro/v1';

	/**
	 * Route base.
	 *
	 * @since 2.5.19
	 *
	 * @var string
	 */
	protected $rest_base = 'zooms/settings';

	/**
	 * Object type.
	 *
	 * @since 2.5.19
	 *
	 * @var string
	 */
	protected $object_type = 'zoom-setting';

	/**
	 * Register routes.
	 *
	 * @since 2.5.19
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_zoom_setting' ),
					'permission_callback' => array( $this, 'get_zoom_setting_permission_check' ),
				),
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'save_zoom_setting' ),
					'permission_callback' => array( $this, 'save_zoom_setting_permission_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/validate',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'validate_settings' ),
					'permission_callback' => array( $this, 'validate_settings_permission_check' ),
				),
			)
		);
	}

	/**
	 * Check if a given request has access to create an item.
	 *
	 * @since 2.5.19
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function save_zoom_setting_permission_check( $request ) {
		return current_user_can( 'edit_zooms' );
	}

	/**
	 * Check if a given request has access to check validate.
	 *
	 * @since 2.5.19
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function validate_settings_permission_check( $request ) {
		return current_user_can( 'edit_zooms' );
	}

	/**
	 * Return validate
	 *
	 * @since 2.5.19
	 *
	 * @return WP_Error|WP_REST_Response
	 */
	public function validate_settings() {
		if ( ! masteriyo_is_zoom_credentials_set() ) {
			return new WP_Error(
				'zoom_credentials_empty',
				__( 'Zoom credentials are not set', 'learning-management-system' ),
				array(
					'status' => 400,
				)
			);
		}

		$valid = masteriyo_validate_zoom_credentials_for_current_user();

		if ( is_wp_error( $valid ) ) {
			return $valid;
		}

		$setting = new Setting();

		return rest_ensure_response( $setting->get_data() );
	}

	/**
	 * Provides the zoom setting data(client_id, client_secret, account_id)  data
	 *
	 * @since 2.5.19
	 *
	 * @return WP_Error|array
	 */
	public function get_zoom_setting() {
		return ( new Setting() )->get_data();
	}

	/**
	 * Add zoom client details to user meta.
	 *
	 * @since 2.5.19
	 *
	 * @param  $request $request Full details about the request.
	 * @return WP_Error|array
	 */
	public function save_zoom_setting( $request ) {
		$client_id     = isset( $request['client_id'] ) ? sanitize_text_field( $request['client_id'] ) : '';
		$client_secret = isset( $request['client_secret'] ) ? sanitize_text_field( $request['client_secret'] ) : '';
		$account_id    = isset( $request['account_id'] ) ? sanitize_text_field( $request['account_id'] ) : '';

		$validate = masteriyo_validate_zoom_credentials( $account_id, $client_id, $client_secret );

		if ( is_wp_error( $validate ) ) {
			return $validate;
		}

		$setting = new Setting();

		$setting->set( 'client_id', $client_id );
		$setting->set( 'client_secret', $client_secret );
		$setting->set( 'account_id', $account_id );
		$setting->save();

		return rest_ensure_response( $setting->get_data() );
	}

	/**
	 * Checks if a given request has access to get items.
	 *
	 * @since 2.5.19
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return true|WP_Error True if the request has read access, WP_Error object otherwise.
	 */
	public function get_zoom_setting_permission_check( $request ) {
		return current_user_can( 'edit_zooms' );
	}

	/**
	 * Get the zooms'schema, conforming to JSON Schema.
	 *
	 * @since 2.5.19
	 *
	 * @return array
	*/
	public function get_item_schema() {
		$schema = array(
			'$schema'    => 'http://json-schema.org/draft-04/schema#',
			'title'      => $this->object_type,
			'type'       => 'object',
			'properties' => array(
				'client_id'     => array(
					'description' => __( 'Client Id', 'learning-management-system' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'client_secret' => array(
					'description' => __( 'Client Secret', 'learning-management-system' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
				'account_id'    => array(
					'description' => __( 'Account Id', 'learning-management-system' ),
					'type'        => 'string',
					'context'     => array( 'view', 'edit' ),
				),
			),
		);

		return $this->add_additional_fields_schema( $schema );
	}
}
