<?php
/**
 * Student registration email to admin class.
 *
 * @package Masteriyo\Emails
 *
 * @since 2.6.10
 */

namespace Masteriyo\Emails\Admin;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

use Masteriyo\Abstracts\Email;

/**
 * Student registration email to admin class. Used for sending new account email.
 *
 * @since 2.6.10
 *
 * @package Masteriyo\Emails
 */
class StudentRegistrationEmailToAdmin extends Email {
	/**
	 * Email method ID.
	 *
	 * @since 2.6.10
	 *
	 * @var string
	 */
	protected $id = 'student-registration/to/admin';

	/**
	 * HTML template path.
	 *
	 * @since 2.6.10
	 *
	 * @var string
	 */
	protected $html_template = 'emails/admin/student-registration.php';

	/**
	 * Send this email.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Models\User|int $user_id
	 */
	public function trigger( $user_id ) {
		$admin_email = get_bloginfo( 'admin_email' );

		// Bail early if order doesn't exist.
		if ( empty( $admin_email ) ) {
			return;
		}

		$student = masteriyo_get_user( $user_id );

		// Bail early if student doesn't exist.
		if ( is_wp_error( $student ) ) {
			return;
		}

		$to_address = explode( ',', $this->format_string( masteriyo_get_setting( 'emails.admin.student_registration.to_address' ) ) ?? $admin_email );
		$this->set_recipients( $to_address );

		$this->set( 'email_heading', $this->get_heading() );
		$this->set( 'student', $student );

		$this->send(
			$this->get_recipients(),
			$this->get_subject(),
			$this->get_content(),
			$this->get_headers(),
			$this->get_attachments()
		);
	}

	/**
	 * Return true if it is enabled.
	 *
	 * @since 2.6.10
	 *
	 * @return bool
	 */
	public function is_enabled() {
		return masteriyo_string_to_bool( masteriyo_get_setting( 'emails.admin.student_registration.enable' ) );
	}

	/**
	 * Return subject.
	 *
	 * @since 2.6.10
	 *
	 * @return string
	 */
	public function get_subject() {
		/**
		 * Filter student registration email subject to admin.
		 *
		 * @since 2.6.10
		 *
		 * @param string $subject.
		 */
		$subject = apply_filters( $this->get_full_id(), masteriyo_get_setting( 'emails.admin.student_registration.subject' ) );

		return $this->format_string( $subject );
	}

	/**
	 * Return heading.
	 *
	 * @since 2.6.10
	 *
	 * @return string
	 */
	public function get_heading() {
		/**
		 * Filter student registration email heading to student.
		 *
		 * @since 2.6.10
		 *
		 * @param string $heading.
		 */
		$heading = apply_filters( $this->get_full_id(), masteriyo_get_setting( 'emails.admin.student_registration.heading' ) );

		return $this->format_string( $heading );
	}

	/**
	 * Return additional content.
	 *
	 * @since 2.6.10
	 *
	 * @return string
	 */
	public function get_additional_content() {

		/**
		 * Filter student registration email additional content to student.
		 *
		 * @since 2.6.10
		 *
		 * @param string $additional_content.
		 */
		$additional_content = apply_filters( $this->get_full_id(), masteriyo_get_setting( 'emails.admin.student_registration.additional_content' ) );

		return $this->format_string( $additional_content );
	}

	/**
	 * Get email content.
	 *
	 * @since 2.6.10
	 *
	 * @return string
	 */
	public function get_content() {
		$content = masteriyo_get_setting( 'emails.admin.student_registration.content' );
		$content = $this->format_string( $content );
		$this->set( 'content', $content );
		return parent::get_content();
	}

	/**
	 * Get placeholders.
	 *
	 * @since 2.6.10
	 *
	 * @return array
	 */
	public function get_placeholders() {
		$placeholders = parent::get_placeholders();

		/** @var \Masteriyo\Models\User $student */
		$student = $this->get( 'student' );

		if ( $student ) {
			$name = trim( sprintf( '%s %s', $student->get_first_name(), $student->get_last_name() ) );

			$placeholders = $placeholders + array(
				'{student_display_name}'    => $student->get_display_name(),
				'{student_first_name}'      => $student->get_first_name(),
				'{student_last_name}'       => $student->get_last_name(),
				'{student_username}'        => $student->get_username(),
				'{student_nicename}'        => $student->get_nicename(),
				'{student_nickname}'        => $student->get_nickname(),
				'{student_name}'            => ! empty( $name ) ? $name : $student->get_display_name(),
				'{student_registered_date}' => gmdate( 'd M Y', $student->get_date_created( 'edit' )->getOffsetTimestamp() ),
			);
		}

		return $placeholders;
	}

	/**
	 * Get the reply_to_name.
	 *
	 * @since 2.8.0
	 *
	 * @return string
	 */
	public function get_reply_to_name() {
		/**
		 * Filter student registration email reply_to_name to admin.
		 *
		 * @since 2.8.0
		 *
		 * @param string $reply_to_name.
		 */
		$reply_to_name = apply_filters( $this->get_full_id() . 'reply_to_name', masteriyo_get_setting( 'emails.admin.student_registration.reply_to_name' ) );
		$reply_to_name = is_string( $reply_to_name ) ? trim( $reply_to_name ) : '';

		return ! empty( $reply_to_name ) ? wp_specialchars_decode( esc_html( $reply_to_name ), ENT_QUOTES ) : parent::get_reply_to_name();
	}

	/**
	 * Get the reply_to_address.
	 *
	 * @since 2.8.0
	 *
	 * @return string
	 */
	public function get_reply_to_address( $reply_to_address = '' ) {
		/**
		 * Filter student registration email reply_to_address to admin.
		 *
		 * @since 2.8.0
		 *
		 * @param string $reply_to_address.
		 */
		$reply_to_address = apply_filters( $this->get_full_id() . 'reply_to_address', masteriyo_get_setting( 'emails.admin.student_registration.reply_to_address' ) );
		$reply_to_address = is_string( $reply_to_address ) ? trim( $reply_to_address ) : '';

		return ! empty( $reply_to_address ) ? sanitize_email( $reply_to_address ) : parent::get_reply_to_address();
	}

	/**
	 * Get the from_name.
	 *
	 * @since 2.8.0
	 *
	 * @return string
	 */
	public function get_from_name() {
		/**
		 * Filter student registration email from_name to admin.
		 *
		 * @since 2.8.0
		 *
		 * @param string $from_name.
		 */
		$from_name = apply_filters( $this->get_full_id() . '_from_name', masteriyo_get_setting( 'emails.admin.student_registration.from_name' ) );
		$from_name = is_string( $from_name ) ? trim( $from_name ) : '';

		return ! empty( $from_name ) ? wp_specialchars_decode( esc_html( $from_name ), ENT_QUOTES ) : parent::get_from_name();
	}

	/**
	 * Get the from_address.
	 *
	 * @since 2.8.0
	 *
	 * @return string
	 */
	public function get_from_address( $from_address = '' ) {
		/**
		 * Filter student registration email from_address to admin.
		 *
		 * @since 2.8.0
		 *
		 * @param string $from_address.
		 */
		$from_address = apply_filters( $this->get_full_id() . '_from_address', masteriyo_get_setting( 'emails.admin.student_registration.from_address' ) );
		$from_address = is_string( $from_address ) ? trim( $from_address ) : '';

		return ! empty( $from_address ) ? sanitize_email( $from_address ) : parent::get_from_address();
	}
}
