<?php
/**
 * User course functions.
 *
 * @since 1.0.0
 * @version 1.0.0
 * @package Masteriyo\Helper
 */
use Masteriyo\Roles;
use Masteriyo\Enums\PostStatus;
use Masteriyo\PostType\PostType;
use Masteriyo\Query\UserCourseQuery;
use Masteriyo\Enums\UserCourseStatus;
use Masteriyo\Query\CourseProgressQuery;
use Masteriyo\Enums\CourseProgressStatus;


/**
 * Get user course.
 *
 * @since 1.0.0
 *
 * @param int $user_course_id User course ID.
 * @return Masteriyo\Models\UserCourse|null
 */
function masteriyo_get_user_course( $user_course_id ) {
	try {
		$user_course = masteriyo( 'user-course' );
		$user_course->set_id( $user_course_id );

		$user_course_repo = masteriyo( 'user-course.store' );
		$user_course_repo->read( $user_course );

		return $user_course;
	} catch ( \Exception $e ) {
		return null;
	}
}

/**
 * Get list of status for user course.
 *
 * @since 1.0.0
 * @deprecated 1.5.3
 *
 * @return array
 */
function masteriyo_get_user_course_statuses() {
	$statuses = array(
		'active' => array(
			'label' => _x( 'Active', 'User Course status', 'learning-management-system' ),
		),
	);

	/**
	 * Filters statuses for user course.
	 *
	 * @since 1.0.0
	 *
	 * @param array $statuses The statuses for user course.
	 */
	return apply_filters( 'masteriyo_user_course_statuses', $statuses );
}

/**
 * Count enrolled users by course or multiple courses.
 *
 * @since 1.0.0
 * @since 1.6.7 Argument $course supports array.
 *
 * @param int|int[] $course Course Id or Course IDS
 *
 * @return integer
 */
function masteriyo_count_enrolled_users( $course ) {
	global $wpdb;

	$count = 0;

	if ( is_array( $course ) ) {
		$course = array_filter( array_map( 'absint', $course ) );
	}

	if ( $wpdb && $course ) {
		$sql = $wpdb->prepare(
			"SELECT COUNT(*) FROM {$wpdb->prefix}masteriyo_user_items WHERE ( status = %s OR status = %s )",
			UserCourseStatus::ACTIVE,
			UserCourseStatus::ENROLLED
		);

		$exclude_users = array_map(
			'absint',
			(array) get_users(
				array(
					'role__in' => array( Roles::ADMIN, Roles::INSTRUCTOR, Roles::MANAGER ),
					'fields'   => 'ID',
				)
			)
		);

		if ( ! empty( $exclude_users ) ) {
			$placeholders = array_fill( 0, count( $exclude_users ), '%d' );
			$sql         .= $wpdb->prepare( ' AND user_id NOT IN (' . implode( ',', $placeholders ) . ')', $exclude_users ); //phpcs:ignore
		}

		if ( is_array( $course ) ) {
			$placeholders = array_fill( 0, count( $course ), '%d' );
			$sql         .= $wpdb->prepare( ' AND item_id IN (' . implode( ',', $placeholders ) . ')', $course ); //phpcs:ignore
		} else {
			$sql .= $wpdb->prepare( ' AND item_id = %d', $course );
		}

		$count = $wpdb->get_var( $sql ); //phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
	}

	/**
	 * Filters enrolled users count for a course.
	 *
	 * @since 1.0.0
	 * @since 1.5.17 Removed third $query parameter.
	 *
	 * @param integer $count The enrolled users count for the given course.
	 * @param int|int[] $course Course ID or Course object.
	 */
	return apply_filters( 'masteriyo_count_enrolled_users', absint( $count ), $course );
}

/**
 * Get enrolled users IDs by course or multiple courses.
 *
 * @since 2.8.0
 *
 * @param int $course Course Id
 *
 * @return array
 */
function masteriyo_get_enrolled_users( $course ) {
	global $wpdb;

	$user_ids = array();

	if ( is_array( $course ) ) {
			$course = array_filter( array_map( 'absint', $course ) );
	}

	if ( $wpdb && $course ) {
			$sql = $wpdb->prepare(
				"SELECT user_id FROM {$wpdb->prefix}masteriyo_user_items WHERE ( status = %s OR status = %s )",
				UserCourseStatus::ACTIVE,
				UserCourseStatus::ENROLLED
			);

			$exclude_users = array_map(
				'absint',
				(array) get_users(
					array(
						'role__in' => array( Roles::ADMIN, Roles::INSTRUCTOR, Roles::MANAGER ),
						'fields'   => 'ID',
					)
				)
			);

		if ( ! empty( $exclude_users ) ) {
				$placeholders = array_fill( 0, count( $exclude_users ), '%d' );
				$sql         .= $wpdb->prepare( ' AND user_id NOT IN (' . implode( ',', $placeholders ) . ')', $exclude_users ); //phpcs:ignore
		}

		if ( is_array( $course ) ) {
				$placeholders = array_fill( 0, count( $course ), '%d' );
				$sql         .= $wpdb->prepare( ' AND item_id IN (' . implode( ',', $placeholders ) . ')', $course ); //phpcs:ignore
		} else {
				$sql .= $wpdb->prepare( ' AND item_id = %d', $course );
		}

			$user_ids = $wpdb->get_col( $sql ); //phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
	}

	/**
	 * Filters enrolled user IDs for a course.
	 *
	 * @since 2.8.0
	 *
	 * @param array   $user_ids Array of user IDs enrolled in the given course.
	 * @param int|int[] $course   Course ID or Course object.
	 */
	return apply_filters( 'masteriyo_get_enrolled_users', array_map( 'absint', $user_ids ), $course );
}


/**
 * Get the number of active courses.
 *
 * @since 1.0.0
 *
 * @param Masteriyo\Models\User|int $user User.
 *
 * @return int
 */
function masteriyo_get_active_courses_count( $user ) {
	global $wpdb;

	$user_id = is_a( $user, 'Masteriyo\Models\User' ) ? $user->get_id() : $user;

	$count = $wpdb->get_var(
		$wpdb->prepare(
			"SELECT COUNT(*) FROM {$wpdb->prefix}masteriyo_user_activities
			WHERE user_id = %d AND activity_type = 'course_progress'
			AND ( activity_status = 'started' OR activity_status = 'progress' )  AND parent_id = 0",
			$user_id
		)
	);

	return $count;
}

/**
 * Get the number of user courses.
 *
 * @since 1.0.0
 * @since 1.6.7 Argument $course supports array.
 * @param int|int[] $course Course id or array of course ids.
 *
 * @return int
 */
function masteriyo_get_user_courses_count_by_course( $course ) {
	global $wpdb;

	$count = 0;

	if ( is_array( $course ) ) {
		$course = array_filter( array_map( 'absint', $course ) );
	}

	if ( $wpdb && $course ) {
		$sql = "SELECT COUNT(*) FROM {$wpdb->prefix}masteriyo_user_items WHERE item_type = 'user_course'";

		if ( is_array( $course ) ) {
			$placeholders = array_fill( 0, count( $course ), '%d' );
			$sql         .= $wpdb->prepare( 'AND item_id IN (' . implode( ',', $placeholders ) . ')', $course ); // phpcs:ignore
		} else {
			$sql .= $wpdb->prepare( 'AND item_id = %d', $course );
		}

		$count = $wpdb->get_var( $sql ); //phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
	}

	/**
	 * Filters user courses count by course.
	 *
	 * @since 1.6.7
	 *
	 * @param integer $count The enrolled users count for the given course.
	 * @param int|int[] $course Course ID or Course object.
	 */
	return apply_filters( 'masteriyo_get_user_courses_count_by_course', absint( $count ), $course );
}

/**
 * Get user/enrolled course by user ID and course ID.
 *
 * @since 1.5.4
 *
 * @param int $user_id User ID.
 * @param int $course_id Course ID.
 * @return Masteriyo\Models\UserCourse
 */
function masteriyo_get_user_course_by_user_and_course( $user_id, $course_id ) {
	$query = new UserCourseQuery(
		array(
			'course_id' => $course_id,
			'user_id'   => $user_id,
		)
	);

	return current( $query->get_user_courses() );
}


/**
 * Retrieves all course IDs for a given user.
 *
 * @since 2.6.5
 *
 * @param int $user_id Optional. User ID. Defaults to 0.
 *
 * @return array Array of course IDs.
 */
function masteriyo_get_all_user_course_ids( $user_id ) {
	global $wpdb;

	$course_ids = array();

	if ( $wpdb ) {
		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT DISTINCT item_id FROM {$wpdb->prefix}masteriyo_user_items WHERE (status = %s OR status = %s) AND user_id = %d",
				array(
					UserCourseStatus::ACTIVE,
					UserCourseStatus::ENROLLED,
					intval( $user_id ),
				)
			)
		);

		if ( $results ) {
			foreach ( $results as $result ) {
				$course_ids[] = $result->item_id;
			}
		}
	}

	return $course_ids;
}

/**
 * Get the enrollment date for a specific user and course.
 *
 * @since 2.7.0
 *
 * @param int $course_id The course ID.
 * @param int $user_id   The user ID.
 *
 * @return string|null The enrollment date or null if not available.
 */
function masteriyo_get_enrollment_date_by_user_course( $course_id, $user_id ) {
	global $wpdb;
	return $wpdb->get_var(
		$wpdb->prepare(
			"SELECT date_start FROM {$wpdb->prefix}masteriyo_user_items WHERE item_id = %d AND user_id = %d AND item_type = 'user_course'",
			$course_id,
			$user_id
		)
	);
}

/**
 * Revoke the enrollment for a specific user and course if it has expired.
 *
 * @since 2.7.0
 *
 * @param int $course_id The course ID.
 * @param int $user_id   The user ID.
 *
 * @return bool True if revoked, false otherwise.
 */
function masteriyo_revoke_enrollment_due_to_expiration_by_user_course( $course_id, $user_id ) {
	global $wpdb;
	return (bool) $wpdb->delete(
		"{$wpdb->prefix}masteriyo_user_items",
		array(
			'user_id'   => $user_id,
			'item_id'   => $course_id,
			'item_type' => 'user_course',
		),
		array(
			'%d',
			'%d',
			'%s',
		)
	);
}

/**
 * Get the remaining time in days before a user loses access to a specific course due to enrollment expiration.
 *
 * @since 2.7.0
 *
 * @param int  $user_id   The user ID for whom to find the remaining time.
 * @param int  $course_id The course ID for which to find the remaining time.
 * @param bool $format    If true, return formatted days as string.
 * @param bool $delete    If true, revoke access if remaining days <= 0.
 *
 * @return int|string|null Remaining time in days or formatted string or null if data is not available or the course doesn't expire.
 */
function masteriyo_get_remaining_time_for_single_course( $user_id, $course_id, $format = false, $delete = false ) {
	$course = masteriyo_get_course( $course_id );

	if ( is_null( $course ) || is_wp_error( $course ) || ! masteriyo_can_start_course( $course_id, $user_id ) ) {
		return null;
	}

	if ( ! $course->get_enrollment_expiration_enabled() ) {
		return null;
	}

	$expiration_duration_days = $course->get_enrollment_expiration_duration();

	if ( 1 > $expiration_duration_days ) {
		return null;
	}

	$enrollment_date = masteriyo_get_enrollment_date_by_user_course( $course_id, $user_id );

	if ( ! $enrollment_date ) {
		return null;
	}

	$current_time          = time();
	$enrollment_start_time = absint( strtotime( $enrollment_date ) );

	$elapsed_time_days   = ( $current_time - $enrollment_start_time ) / DAY_IN_SECONDS;
	$remaining_time_days = $expiration_duration_days - $elapsed_time_days;

	// If the remaining time is <= 0 and delete is true, then revoke access and return null.
	if ( $remaining_time_days <= 0 && $delete ) {
		masteriyo_revoke_enrollment_due_to_expiration_by_user_course( $course_id, $user_id );
		return null;
	}

	$rounded_remaining_days = absint( round( $remaining_time_days ) );

	if ( $rounded_remaining_days <= 0 ) {
		return null;
	}

	if ( $format ) {
		/* translators: %d: Number of remaining days */
		return 1 === $rounded_remaining_days ? __( '1 Day', 'learning-management-system' ) : sprintf( __( '%d Days', 'learning-management-system' ), $rounded_remaining_days );
	}

	return $rounded_remaining_days;
}

if ( ! function_exists( 'masteriyo_count_all_enrolled_users' ) ) {
	/**
	 * Count total enrolled users from all courses.
	 *
	 * @since 1.6.16
	 *
	 * @param int|WP_User|Masteriyo\Database\Model $user User ID, WP_User object, or Masteriyo\Database\Model object.
	 *
	 * @return integer
	 */
	function masteriyo_count_all_enrolled_users( $user ) {
		$total_count = 0;

		$user = masteriyo_get_user( $user );

		if ( is_null( $user ) || is_wp_error( $user ) ) {
			return $total_count;
		}

		// Get all courses.
		$all_courses = get_posts(
			array(
				'post_type'      => PostType::COURSE,
				'post_status'    => PostStatus::PUBLISH,
				'author'         => $user->get_id(),
				'posts_per_page' => -1,
				'fields'         => 'ids',
			)
		);

		// Iterate through each course and count enrolled users.
		foreach ( $all_courses as $course_id ) {
				$total_count += masteriyo_count_enrolled_users( $course_id );
		}

		return $total_count;
	}
}

if ( ! function_exists( 'masteriyo_count_user_courses' ) ) {
	/**
	 * Get the count of courses created by a user.
	 *
	 * @since 1.6.16
	 *
	 * @param int|WP_User|Masteriyo\Database\Model $user User ID, WP_User object, or Masteriyo\Database\Model object.
	 *
	 * @return int The count of courses created by the user.
	 */
	function masteriyo_count_user_courses( $user ) {
		$user = masteriyo_get_user( $user );

		if ( is_null( $user ) || is_wp_error( $user ) ) {
			return 0;
		}

		$query = new WP_Query(
			array(
				'post_type'      => PostType::COURSE,
				'post_status'    => PostStatus::PUBLISH,
				'author'         => $user->get_id(),
				'posts_per_page' => -1,
				'fields'         => 'ids',
			)
		);

		return $query->found_posts;

	}
}

if ( ! function_exists( 'masteriyo_get_user_enrolled_courses' ) ) {
	/**
	 * Retrieves the enrolled courses of a user.
	 *
	 * @since 2.6.8
	 *
	 * @param int|WP_User|Masteriyo\Database\Model $user User ID, WP_User object, or Masteriyo\Database\Model object.
	 * @param int $limit The maximum number of courses to retrieve. Default is -1 (unlimited).
	 *
	 * @return array The user's enrolled courses.
	 */
	function masteriyo_get_user_enrolled_courses( $user, $limit = -1 ) {
		$user = masteriyo_get_user( $user );

		if ( is_null( $user ) || is_wp_error( $user ) ) {
			return array();
		}

		$query = new CourseProgressQuery(
			array(
				'user_id'  => $user->get_id(),
				'per_page' => $limit,
				'status'   => array( CourseProgressStatus::STARTED, CourseProgressStatus::PROGRESS, CourseProgressStatus::COMPLETED ),
			)
		);

		$progresses = $query->get_course_progress();

		$enrolled_courses = array_filter(
			array_map(
				function( $progress ) {
					$course = masteriyo_get_course( $progress->get_course_id() );

					if ( is_null( $course ) ) {
						return null;
					}

					$course->progress = $progress;
					return $course;
				},
				$progresses
			)
		);

		return $enrolled_courses;
	}
}

if ( ! function_exists( 'masteriyo_is_user_already_enrolled' ) ) {
	/**
	 * Checks if a user is enrolled in a specific course, optionally filtering by enrollment status.
	 *
	 * @since 1.8.3
	 *
	 * @param int         $user_id   The ID of the user.
	 * @param int         $course_id The ID of the course.
	 * @param string|null $status    Optional. The enrollment status to check ('active', 'inactive'.). Default null.
	 *
	 * @return bool True if the user is enrolled with the specified status (if provided), false otherwise.
	 */
	function masteriyo_is_user_already_enrolled( $user_id, $course_id, $status = null ) {
		global $wpdb;

		if ( ! $wpdb || ! $user_id || ! $course_id ) {
			return false;
		}

		$query = "SELECT COUNT(*) FROM {$wpdb->prefix}masteriyo_user_items WHERE user_id = %d AND item_id = %d AND item_type = 'user_course'";
		$args  = array( $user_id, $course_id );

		if ( ! is_null( $status ) ) {
			$query .= ' AND status = %s';
			$args[] = $status;
		}

		$query .= ' LIMIT 1';

		$count = $wpdb->get_var( $wpdb->prepare( $query, $args ) ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
		return $count > 0;
	}
}

if ( ! function_exists( 'masteriyo_is_request_from_account_dashboard' ) ) {
	/**
	 * Determines if the request is from the account dashboard.
	 *
	 * @since 1.14.2 [Free]
	 *
	 * @param WP_REST_Request|null $request Optional. The request object. Defaults to current HTTP request.
	 *
	 * @return bool True if the request is from the account dashboard, false otherwise.
	 */
	function masteriyo_is_request_from_account_dashboard( $request = null ) {
		$request = $request ?? masteriyo_current_http_request();

		if ( ! $request instanceof \WP_REST_Request ) {
			return false;
		}

		return masteriyo_string_to_bool( $request['from_account_dashboard'] ) ?? false;
	}
}

if ( ! function_exists( 'masteriyo_get_user_progress_course_ids' ) ) {

	/**
	 * Retrieves an array of course IDs for a given user filtered by the specified course status.
	 *
	 * @since 1.14.2 [Free]
	 *
	 * @param Masteriyo\Models\User|int|null $user Optional. User object or ID. Defaults to the current user.
	 * @param string $course_status Optional. The status of the course. Defaults to 'progress'.
	 *
	 * @return array The array of course IDs matching the specified status for the user.
	 */
	function masteriyo_get_user_course_ids_by_course_status( $user = null, $course_status = CourseProgressStatus::PROGRESS ) {

		$user_id = is_a( $user, 'Masteriyo\Models\User' ) ? $user->get_id() : absint( $user ) ?? get_current_user_id();

		if ( ! $user_id ) {
			return array();
		}

		global $wpdb;

		$courses_ids = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT a.item_id
					FROM {$wpdb->prefix}masteriyo_user_activities a
					WHERE a.user_id = %d
					AND a.activity_type = %s
					AND a.activity_status = %s
					AND a.item_id IN (
						SELECT b.item_id FROM {$wpdb->prefix}masteriyo_user_items b WHERE b.status = %s
					)
					AND a.item_id IN (
						SELECT c.ID FROM {$wpdb->prefix}posts c WHERE c.post_type = %s AND c.post_status = %s
					) ORDER BY a.item_id DESC",
				array(
					absint( $user_id ),
					'course_progress',
					$course_status,
					UserCourseStatus::ACTIVE,
					PostType::COURSE,
					PostStatus::PUBLISH,
				)
			)
		);

		return $courses_ids;
	}
}

if ( ! function_exists( 'masteriyo_get_user_courses_count_by_course_status' ) ) {

	/**
	 * Get the count of user courses by course status.
	 *
	 * Retrieves the number of courses for a given user based on the specified course status.
	 *
	 * @since 1.14.2 [Free]
	 *
	 * @param Masteriyo\Models\User|int|null $user Optional. User object or ID. Defaults to the current user.
	 * @param string $course_status Optional. The status of the course. Defaults to 'progress'.
	 *
	 * @return int The count of courses matching the specified status for the user.
	 */
	function masteriyo_get_user_courses_count_by_course_status( $user = null, $course_status = CourseProgressStatus::PROGRESS ) {
		$user_id = is_a( $user, 'Masteriyo\Models\User' ) ? $user->get_id() : absint( $user ) ?? get_current_user_id();

		if ( ! $user_id ) {
			return 0;
		}

		global $wpdb;

		$courses_count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*)
					FROM {$wpdb->prefix}masteriyo_user_activities a
					WHERE a.user_id = %d
					AND a.activity_type = %s
					AND a.activity_status = %s
					AND a.item_id IN (
						SELECT b.item_id FROM {$wpdb->prefix}masteriyo_user_items b WHERE b.status = %s
					)
					AND a.item_id IN (
						SELECT c.ID FROM {$wpdb->prefix}posts c WHERE c.post_type = %s AND c.post_status = %s
					) ORDER BY a.item_id DESC",
				array(
					absint( $user_id ),
					'course_progress',
					$course_status,
					UserCourseStatus::ACTIVE,
					PostType::COURSE,
					PostStatus::PUBLISH,
				)
			)
		);

		return $courses_count ? absint( $courses_count ) : 0;
	}
}

if ( ! function_exists( 'masteriyo_get_user_enrolled_courses_count' ) ) {

	/**
	 * Retrieves the number of courses in which a user is enrolled.
	 *
	 * @since 1.14.2 [Free]
	 *
	 * @param int|WP_User|Masteriyo\Database\Model $user User ID, WP_User object, or Masteriyo\Database\Model object.
	 *
	 * @return int The number of enrolled courses for the user.
	 */
	function masteriyo_get_user_enrolled_courses_count( $user = null ) {
		$user_id = is_a( $user, 'Masteriyo\Models\User' ) ? $user->get_id() : absint( $user ) ?? get_current_user_id();

		if ( ! $user_id ) {
			return 0;
		}

		global $wpdb;

		$courses_count = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*)
					FROM {$wpdb->prefix}masteriyo_user_activities a
					WHERE a.user_id = %d
					AND a.activity_type = %s
					AND a.item_id IN (
						SELECT b.item_id FROM {$wpdb->prefix}masteriyo_user_items b WHERE b.status = %s
					)
					AND a.item_id IN (
						SELECT c.ID FROM {$wpdb->prefix}posts c WHERE c.post_type = %s AND c.post_status = %s
					) ORDER BY a.item_id DESC",
				array(
					absint( $user_id ),
					'course_progress',
					UserCourseStatus::ACTIVE,
					PostType::COURSE,
					PostStatus::PUBLISH,
				)
			)
		);

		return $courses_count ? absint( $courses_count ) : 0;
	}
}

if ( ! function_exists( 'masteriyo_delete_user_course_assignment_submissions' ) ) {
	/**
	 * Remove assignment submissions of a course for the user.
	 *
	 * @since 2.16.0
	 *
	 * @param int $course_id The course ID.
	 * @param int $user_id   The user ID.
	 */
	function masteriyo_delete_user_course_assignment_submissions( $course_id, $user_id ) {
		$course_id = absint( $course_id );
		$user_id   = absint( $user_id );

		if ( ! $course_id || ! $user_id ) {
			return;
		}

		global $wpdb;

		$args           = array(
			'post_type'      => PostType::ASSIGNMENT,
			'post_status'    => 'any',
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'meta_query'     => array(
				'relation' => 'AND',
				array(
					'key'     => '_course_id',
					'value'   => $course_id,
					'compare' => '=',
				),
			),
		);
		$query          = new \WP_Query( $args );
		$assignment_ids = $query->posts;

		if ( empty( $assignment_ids ) ) {
			return;
		}

		$ids_str = implode( ', ', array_fill( 0, count( $assignment_ids ), '%d' ) );
		$sql     = "DELETE FROM {$wpdb->posts} WHERE post_author = %d AND post_parent IN ({$ids_str})";

		$wpdb->query( $wpdb->prepare( $sql, array_merge( array( $user_id ), $assignment_ids ) ) ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
	}
}

if ( ! function_exists( 'masteriyo_delete_user_course_gradebooks' ) ) {
	/**
	 * Remove assignment submissions of a course for the current user.
	 *
	 * @since 2.7.0
	 *
	 * @param int $course_id The course ID.
	 * @param int $user_id  The user ID.
	 */
	function masteriyo_delete_user_course_gradebooks( $course_id, $user_id ) {
		$course_id = absint( $course_id );
		$user_id   = absint( $user_id );

		if ( ! $course_id || ! $user_id ) {
			return;
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'masteriyo_gradebook_results';

		if ( $wpdb->get_var( "SHOW TABLES LIKE '$table_name'" ) === $table_name ) {  // phpcs:ignore
			$gradebook_id = absint(
				$wpdb->get_var(
					$wpdb->prepare(
						"SELECT id FROM {$wpdb->prefix}masteriyo_gradebook_results
					WHERE item_id = %d
					AND user_id = %d
					AND item_type = 'course'",
						$course_id,
						$user_id
					)
				)
			);

			$wpdb->delete(
				"{$wpdb->prefix}masteriyo_gradebook_results",
				array(
					'parent_id' => $gradebook_id,
				)
			);

			$wpdb->delete(
				"{$wpdb->prefix}masteriyo_gradebook_results",
				array(
					'id' => $gradebook_id,
				)
			);
		}
	}
}
