<?php
/**
 * Payment retry model.
 *
 * @since 2.6.10
 *
 * @package Masteriyo\Models
 */

namespace Masteriyo\Models;

use Masteriyo\Database\Model;
use Masteriyo\Repository\RepositoryInterface;

defined( 'ABSPATH' ) || exit;

/**
 * Payment retry model (custom table).
 *
 * @since 2.6.10
 */
class PaymentRetry extends Model {

	/**
	 * This is the name of this object type.
	 *
	 * @since 2.6.10
	 *
	 * @var string
	 */
	protected $object_type = 'payment-retry';

	/**
	 * Cache group.
	 *
	 * @since 2.6.10
	 *
	 * @var string
	 */
	protected $cache_group = 'payment-retries';

	/**
	 * Stores payment retry data.
	 *
	 * @since 2.6.10
	 *
	 * @var array
	 */
	protected $data = array(
		'order_id'   => '',
		'status'     => false,
		'rules'      => '',
		'created_at' => null,
	);

	/**
	 * Constructor.
	 *
	 * @since 2.6.10
	 *
	 * @param RepositoryInterface $payment_retry_repository Payment retry Repository,
	 */
	public function __construct( RepositoryInterface $payment_retry_repository ) {
		$this->repository = $payment_retry_repository;
	}

	/*
	|--------------------------------------------------------------------------
	| Non-CRUD Getters
	|--------------------------------------------------------------------------
	*/

	/**
	 * Return order object.
	 *
	 * @since 2.6.10
	 *
	 * @return \Masteriyo\Models\Order\Order|null
	 */
	public function get_order() {
		return masteriyo_get_order( $this->get_order_id() );
	}

	/**
	 * Return table name.
	 *
	 * @since 2.6.10
	 *
	 * @return string
	 */
	public function get_table_name() {
		global $wpdb;

		return "{$wpdb->prefix}masteriyo_payment_retries";
	}

	/*
	|--------------------------------------------------------------------------
	| Getters
	|--------------------------------------------------------------------------
	*/

	/**
	 * Return order ID.
	 *
	 * @since  2.6.10
	 * @param  string $context What the value is for. Valid values are view and edit.
	 *
	 * @return int
	 */
	public function get_order_id( $context = 'view' ) {
		return $this->get_prop( 'order_id', $context );
	}

	/**
	 * Return status.
	 *
	 * @since  2.6.10
	 * @param  string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 */
	public function get_status( $context = 'view' ) {
		return $this->get_prop( 'status', $context );
	}

	/**
	 * Return rules.
	 *
	 * @since  2.6.10
	 * @param  string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 */
	public function get_rules( $context = 'view' ) {
		return $this->get_prop( 'rules', $context );
	}

	/**
	 * Return created at.
	 *
	 * @since  2.6.10
	 * @param  string $context What the value is for. Valid values are view and edit.
	 *
	 * @return \Masteriyo\DateTime|null object if the date is set or null if there is no date.
	 */
	public function get_created_at( $context = 'view' ) {
		return $this->get_prop( 'created_at', $context );
	}

	/**
	 * Return modified at.
	 *
	 * @since  2.6.10
	 * @param  string $context What the value is for. Valid values are view and edit.
	 *
	 * @return \Masteriyo\DateTime|null object if the date is set or null if there is no date.
	 */
	public function get_modified_at( $context = 'view' ) {
		return $this->get_prop( 'modified_at', $context );
	}

	/*
	|--------------------------------------------------------------------------
	| Setters
	|--------------------------------------------------------------------------
	*/

	/**
	 * Set order ID.
	 *
	 * @since  2.6.10
	 * @param int $order_id Order ID.
	 */
	public function set_order_id( $order_id ) {
		return $this->set_prop( 'order_id', absint( $order_id ) );
	}

	/**
	 * Set status
	 *
	 * @since  2.6.10
	 * @param string $status Status
	 */
	public function set_status( $status ) {
		return $this->set_prop( 'status', $status );
	}

	/**
	 * Set rules.
	 *
	 * @since  2.6.10
	 * @param string $rules Rules
	 */
	public function set_rules( $rules ) {
		return $this->set_prop( 'rules', $rules );
	}

	/**
	 * Set created at.
	 *
	 * @since  2.6.10
	 * @param string|integer|null $created_at UTC timestamp, or ISO 8601 DateTime. If the DateTime string has no timezone or offset, WordPress site timezone will be assumed. Null if their is no date.
	 */
	public function set_created_at( $created_at ) {
		return $this->set_date_prop( 'created_at', $created_at );
	}

	/**
	 * Set modified at.
	 *
	 * @since  2.6.10
	 * @param string|integer|null $modified_at UTC timestamp, or ISO 8601 DateTime. If the DateTime string has no timezone or offset, WordPress site timezone will be assumed. Null if their is no date.
	 */
	public function set_modified_at( $modified_at ) {
		return $this->set_date_prop( 'modified_at', $modified_at );
	}
}
