<?php
/**
 * Subscription class controller.
 *
 * @since 2.6.10
 *
 * @package Masteriyo\RestApi\Controllers\Version1;
 */

namespace Masteriyo\Pro\Controllers;

defined( 'ABSPATH' ) || exit;

use Masteriyo\Enums\OrderStatus;
use Masteriyo\Enums\CourseBillingPeriod;
use Masteriyo\Pro\Models\Subscription;
use Masteriyo\RestApi\Controllers\Version1\OrdersController;
use Masteriyo\Pro\Enums\SubscriptionStatus;
use Masteriyo\Helper\Permission;
use Masteriyo\Exceptions\RestException;
use Masteriyo\ModelException;
use Masteriyo\PostType\PostType;

/**
 * SubscriptionsController class.
 */
class SubscriptionsController extends OrdersController {

	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'masteriyo/pro/v1';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'subscriptions';

	/**
	 * Object type.
	 *
	 * @var string
	 */
	protected $object_type = 'subscription';

	/**
	 * Post type.
	 *
	 * @var string
	 */
	protected $post_type = PostType::SUBSCRIPTION;

	/**
	 * If object is hierarchical.
	 *
	 * @var bool
	 */
	protected $hierarchical = true;

	/**
	 * Permission class.
	 *
	 * @since 2.6.10
	 *
	 * @var \Masteriyo\Helper\Permission;
	 */
	protected $permission = null;

	/**
	 * Constructor.
	 *
	 * @since 2.6.10
	 *
	 * @param Permission $permission Permission object.
	 */
	public function __construct( Permission $permission = null ) {
		$this->permission = $permission;
	}

	/**
	 * Register routes.
	 *
	 * @since 2.6.10
	 *
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => $this->get_collection_params(),
				),
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::CREATABLE ),
				),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/(?P<id>[\d]+)',
			array(
				'args'   => array(
					'id' => array(
						'description' => __( 'Unique identifier for the resource.', 'learning-management-system' ),
						'type'        => 'integer',
					),
				),
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_item' ),
					'permission_callback' => array( $this, 'get_item_permissions_check' ),
					'args'                => array(
						'context' => $this->get_context_param(
							array(
								'default' => 'view',
							)
						),
					),
				),
				array(
					'methods'             => \WP_REST_Server::EDITABLE,
					'callback'            => array( $this, 'update_item' ),
					'permission_callback' => array( $this, 'update_item_permissions_check' ),
					'args'                => $this->get_endpoint_args_for_item_schema( \WP_REST_Server::EDITABLE ),
				),
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_item' ),
					'permission_callback' => array( $this, 'delete_item_permissions_check' ),
					'args'                => array(
						'force' => array(
							'default'     => false,
							'description' => __( 'Whether to bypass trash and force deletion.', 'learning-management-system' ),
							'type'        => 'boolean',
						),
					),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
			)
		);

		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base . '/delete',
			array(
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_items' ),
					'permission_callback' => array( $this, 'delete_items_permissions_check' ),
					'args'                => array(
						'ids'   => array(
							'required'    => true,
							'description' => __( 'Subscription IDs.', 'learning-management-system' ),
							'type'        => 'array',
						),
						'force' => array(
							'default'     => false,
							'description' => __( 'Whether to bypass trash and force deletion.', 'learning-management-system' ),
							'type'        => 'boolean',
						),
					),
				),
			)
		);
	}

	/**
	 * Get object.
	 *
	 * @param  int|WP_Post $id Object ID.
	 * @return object Model object or WP_Error object.
	 */
	protected function get_object( $id ) {
		try {
			$id           = $id instanceof \WP_Post ? $id->ID : $id;
			$id           = $id instanceof Subscription ? $id->get_id() : $id;
			$subscription = masteriyo( 'subscription' );
			$subscription->set_id( $id );
			$subscription_repo = masteriyo( 'subscription.store' );
			$subscription_repo->read( $subscription );
		} catch ( \Exception $e ) {
			return false;
		}

		return $subscription;
	}

	/**
	 * Prepares the object for the REST response.
	 *
	 * @since  2.6.10
	 *
	 * @param  Masteriyo\Database\Model $object  Model object.
	 * @param  WP_REST_Request $request Request object.
	 *
	 * @return WP_Error|WP_REST_Response Response object on success, or WP_Error object on failure.
	 */
	protected function prepare_object_for_response( $object, $request ) {
		$context = ! empty( $request['context'] ) ? $request['context'] : 'view';
		$data    = $this->get_subscription_data( $object, $context );

		$data     = $this->add_additional_fields_to_object( $data, $request );
		$data     = $this->filter_response_by_context( $data, $context );
		$response = rest_ensure_response( $data );
		$response->add_links( $this->prepare_links( $object, $request ) );

		/**
		 * Filter the data for a response.
		 *
		 * The dynamic portion of the hook name, $this->object_type,
		 * refers to object type being prepared for the response.
		 *
		 * @since 2.6.10
		 *
		 * @param WP_REST_Response $response The response object.
		 * @param Masteriyo\Database\Model $object   Object data.
		 * @param WP_REST_Request  $request  Request object.
		 */
		return apply_filters( "masteriyo_rest_prepare_{$this->object_type}_object", $response, $object, $request );
	}

	/**
	 * Process objects collection.
	 *
	 * @since 2.6.10
	 *
	 * @param array $objects Subscriptions data.
	 * @param array $query_args Query arguments.
	 * @param array $query_results Subscriptions query result data.
	 *
	 * @return array
	 */
	protected function process_objects_collection( $objects, $query_args, $query_results ) {
		return array(
			'data' => $objects,
			'meta' => array(
				'total'               => $query_results['total'],
				'pages'               => $query_results['pages'],
				'current_page'        => $query_args['paged'],
				'per_page'            => $query_args['posts_per_page'],
				'subscriptions_count' => $this->get_subscriptions_count(),
			),
		);
	}

	/**
	 * Get subscription data.
	 *
	 * @param \Masteriyo\Pro\Models\Subscription  $subscription Subscription instance.
	 * @param string $context Request context.
	 *                        Options: 'view' and 'edit'.
	 *
	 * @return array
	 */
	protected function get_subscription_data( $subscription, $context = 'view' ) {
		$customer      = masteriyo_get_user( $subscription->get_customer_id( $context ) );
		$customer_info = null;

		if ( ! is_wp_error( $customer ) ) {
			$customer_info = array(
				'id'           => $customer->get_id(),
				'display_name' => $customer->get_display_name(),
				'avatar_url'   => $customer->get_avatar_url(),
				'email'        => $customer->get_email(),
			);
		}

		$data = array(
			'id'                         => $subscription->get_id(),
			'permalink'                  => $subscription->get_permalink(),
			'status'                     => $subscription->get_status( $context ),
			'total'                      => $subscription->get_total( $context ),
			'formatted_total'            => $subscription->get_rest_formatted_total( $context ),
			'currency'                   => $subscription->get_currency( $context ),
			'currency_symbol'            => html_entity_decode( masteriyo_get_currency_symbol( $subscription->get_currency( $context ) ) ),
			'expiry_date'                => $subscription->get_expiry_date( $context ),
			'date_created'               => masteriyo_rest_prepare_date_response( $subscription->get_date_created( $context ) ),
			'date_modified'              => masteriyo_rest_prepare_date_response( $subscription->get_date_modified( $context ) ),
			'customer_id'                => $subscription->get_customer_id( $context ),
			'customer'                   => $customer_info,
			'payment_method'             => $subscription->get_payment_method( $context ),
			'payment_method_title'       => $subscription->get_payment_method_title( $context ),
			'transaction_id'             => $subscription->get_transaction_id( $context ),
			'date_paid'                  => $subscription->get_date_paid( $context ),
			'date_completed'             => $subscription->get_date_completed( $context ),
			'created_via'                => $subscription->get_created_via( $context ),
			'customer_ip_address'        => $subscription->get_customer_ip_address( $context ),
			'customer_user_agent'        => $subscription->get_customer_user_agent( $context ),
			'version'                    => $subscription->get_version( $context ),
			'order_key'                  => $subscription->get_order_key( $context ),
			'customer_note'              => $subscription->get_customer_note( $context ),
			'cart_hash'                  => $subscription->get_cart_hash( $context ),
			'billing'                    => array(
				'first_name' => $subscription->get_billing_first_name(),
				'last_name'  => $subscription->get_billing_last_name(),
				'company'    => $subscription->get_billing_company(),
				'address_1'  => $subscription->get_billing_address_1(),
				'address_2'  => $subscription->get_billing_address_2(),
				'city'       => $subscription->get_billing_city(),
				'postcode'   => $subscription->get_billing_postcode(),
				'country'    => $subscription->get_billing_country(),
				'state'      => $subscription->get_billing_state(),
				'email'      => $subscription->get_billing_email(),
				'phone'      => $subscription->get_billing_phone(),
			),
			'course_lines'               => $this->get_order_item_course( $subscription->get_items(), $context ),
			'billing_period'             => $subscription->get_billing_period( $context ),
			'billing_interval'           => $subscription->get_billing_interval( $context ),
			'billing_expire_after'       => $subscription->get_billing_expire_after( $context ),
			'suspension_count'           => $subscription->get_suspension_count( $context ),
			'requires_manual_renewal'    => $subscription->get_requires_manual_renewal( $context ),
			'cancelled_email_sent'       => $subscription->get_cancelled_email_sent( $context ),
			'trial_period'               => $subscription->get_trial_period( $context ),
			'switch_data'                => $subscription->get_switch_data( $context ),
			'schedule_trial_end'         => masteriyo_rest_prepare_date_response( $subscription->get_schedule_trial_end( $context ) ),
			'schedule_next_payment'      => masteriyo_rest_prepare_date_response( $subscription->get_schedule_next_payment( $context ) ),
			'schedule_cancelled'         => masteriyo_rest_prepare_date_response( $subscription->get_schedule_cancelled( $context ) ),
			'schedule_end'               => masteriyo_rest_prepare_date_response( $subscription->get_schedule_end( $context ) ),
			'schedule_payment_retry'     => $subscription->get_schedule_payment_retry( $context ),
			'schedule_start'             => masteriyo_rest_prepare_date_response( $subscription->get_schedule_start( $context ) ),
			'subscription_id'            => $subscription->get_subscription_id( $context ),
			'parent_id'                  => $subscription->get_parent_id( $context ),
			'recurring_amount'           => $subscription->get_recurring_amount( $context ),
			'recurring_amount_formatted' => $subscription->get_parent_id( $context ),
		);

		/**
		 * Filter Subscription rest response data.
		 *
		 * @since 2.6.100
		 *
		 * @param array $data Subscription data.
		 * @param \Masteriyo\Pro\Models\Subscription $subscription Subscription object.
		 * @param string $context What the value is for. Valid values are view and edit.
		 * @param \Masteriyo\Pro\Controllers\SubscriptionsController $controller REST Subscriptions controller object.
		 */
		return apply_filters( "masteriyo_rest_response_{$this->object_type}_data", $data, $subscription, $context, $this );
	}

	/**
	 * Prepare objects query.
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @since  2.6.10
	 * @return array
	 */
	protected function prepare_objects_query( $request ) {
		$args = parent::prepare_objects_query( $request );

		// Set subscription status.
		$args['post_status'] = $request['status'];

		if ( masteriyo_is_current_user_admin() || masteriyo_is_current_user_manager() ) {
			if ( ! empty( $request['customer'] ) ) {
				$args['meta_query'] = array(
					'relation' => 'AND',
					array(
						'key'     => '_customer_id',
						'value'   => absint( $request['customer'] ),
						'compare' => '=',
					),
				);
			}
		} else {
			$args['meta_query'] = array(
				'relation' => 'AND',
				array(
					'key'     => '_customer_id',
					'value'   => get_current_user_id(),
					'compare' => '=',
				),
			);
		}

		if ( ! empty( $request['created_via'] ) ) {
			if ( empty( $args['meta_query'] ) ) {
				$args['meta_query'] = array(
					'relation' => 'AND',
				);
			}

			$args['meta_query'][] = array(
				'key'     => '_created_via',
				'value'   => sanitize_text_field( $request['created_via'] ),
				'compare' => '=',
			);
		}

		return $args;
	}

	/**
	 * Get the subscriptions' schema, conforming to JSON Schema.
	 *
	 * @since 2.6.10
	 *
	 * @return array
	 */
	public function get_item_schema() {
		$parent_schema = parent::get_item_schema();

		$schema = masteriyo_parse_args(
			$parent_schema,
			array(
				'properties' => array(
					'billing_period'          => array(
						'description' => __( 'Billing period', 'learning-management-system' ),
						'type'        => 'string',
						'enum'        => CourseBillingPeriod::all(),
						'context'     => array( 'view', 'edit' ),
					),
					'billing_interval'        => array(
						'description' => __( 'Billing period', 'learning-management-system' ),
						'type'        => 'integer',
						'max'         => 6,
						'min'         => 1,
						'context'     => array( 'view', 'edit' ),
					),
					'suspension_count'        => array(
						'description' => __( 'Billing period', 'learning-management-system' ),
						'type'        => 'integer',
						'readonly'    => true,
						'context'     => array( 'view', 'edit' ),
					),
					'requires_manual_renewal' => array(
						'description' => __( 'Requires manual renewal', 'learning-management-system' ),
						'type'        => 'boolean',
						'context'     => array( 'view', 'edit' ),
					),
					'cancelled_email_sent'    => array(
						'description' => __( 'Cancelled email sent', 'learning-management-system' ),
						'type'        => 'boolean',
						'readonly'    => true,
						'context'     => array( 'view', 'edit' ),
					),
					'trial_period'            => array(
						'description' => __( 'Trial period', 'learning-management-system' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'schedule_trial_end'      => array(
						'description' => __( 'Schedule trial end as ISO 8601 GMT format', 'learning-management-system' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'schedule_next_payment'   => array(
						'description' => __( 'Schedule next payment as ISO 8601 GMT format', 'learning-management-system' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'schedule_cancelled'      => array(
						'description' => __( 'Schedule cancelled as ISO 8601 GMT format', 'learning-management-system' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'schedule_end'            => array(
						'description' => __( 'Scheduled end as ISO 8601 GMT format', 'learning-management-system' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'schedule_payment_retry'  => array(
						'description' => __( 'Scheduled payment retry as ISO 8601 GMT format', 'learning-management-system' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'schedule_start'          => array(
						'description' => __( 'Scheduled start', 'learning-management-system' ),
						'type'        => 'string',
						'context'     => array( 'view', 'edit' ),
					),
					'status'                  => array(
						'description' => __( 'Subscription status', 'learning-management-system' ),
						'type'        => 'string',
						'default'     => 'pending',
						'enum'        => array_merge( array_keys( SubscriptionStatus::list() ), array( 'any', 'active' ) ),
						'context'     => array( 'view', 'edit' ),
					),
				),
			)
		);

		return $this->add_additional_fields_schema( $schema );
	}

	/**
	 * Prepare a single subscription for create or update.
	 *
	 * @since 2.6.10
	 *
	 * @param WP_REST_Request $request Request object.
	 * @param bool            $creating If is creating a new object.
	 *
	 * @return WP_Error|Masteriyo\Database\Model
	 */
	protected function prepare_object_for_database( $request, $creating = false ) {
		$id = isset( $request['id'] ) ? absint( $request['id'] ) : 0;
		/** @var \Masteriyo\Pro\Models\Subscription */
		$subscription = masteriyo( 'subscription' );

		if ( 0 !== $id ) {
			$subscription->set_id( $id );
			$subscription_repo = masteriyo( 'subscription.store' );
			$subscription_repo->read( $subscription );
		} else {
			$order = parent::prepare_object_for_database( $request, $creating );

			if ( ! is_wp_error( $order ) ) {
				if ( empty( $order->get_created_via() ) ) {
					$order->set_created_via( 'rest-api' );
				}

				$order->set_prices_include_tax( 'yes' === get_option( 'masteriyo_prices_include_tax' ) );
				$order->set_status( OrderStatus::COMPLETED );
				$order->calculate_totals();

				$new_order = masteriyo_get_order( $order );

				if ( $new_order ) {
					$subscription->set_parent_id( $new_order->get_id() );
				}
			}
		}

		// Currency.
		if ( isset( $request['currency'] ) ) {
			$subscription->set_currency( $request['currency'] );
		}

		// Customer ID.
		if ( isset( $request['customer_id'] ) ) {
			$subscription->set_customer_id( $request['customer_id'] );
		}

		// Set payment method.
		if ( isset( $request['payment_method'] ) ) {
			$subscription->set_payment_method( $request['payment_method'] );
		}

		// Set payment method title.
		if ( isset( $request['payment_method_title'] ) ) {
			$subscription->set_payment_method_title( $request['payment_method_title'] );
		}

		// Set transaction ID.
		if ( isset( $request['transaction_id'] ) ) {
			$subscription->set_transaction_id( $request['transaction_id'] );
		}

		// Set customer note.
		if ( isset( $request['customer_note'] ) ) {
			$subscription->set_customer_note( $request['customer_note'] );
		}

		// Set subscription status.
		if ( isset( $request['status'] ) ) {
			$subscription->set_status( $request['status'] );
		}

		// Set status as paid.
		if ( $request['set_paid'] ) {
			$subscription->set_status( SubscriptionStatus::COMPLETED );
		}

		// Set created_via.
		if ( $request['created_via'] ) {
			$subscription->set_created_via( $request['created_via'] );
		}

		// Add course items.
		if ( isset( $request['course_lines'] ) ) {
			foreach ( $request['course_lines'] as $course_line ) {
				$this->set_item( $subscription, 'course_lines', $course_line );
			}
		}

		// Set customer IP address.
		$subscription->set_customer_ip_address( masteriyo_get_current_ip_address() );

		// Subscription billing details.
		if ( isset( $request['billing']['first_name'] ) ) {
			$subscription->set_billing_first_name( $request['billing']['first_name'] );
		}

		if ( isset( $request['billing']['last_name'] ) ) {
			$subscription->set_billing_last_name( $request['billing']['last_name'] );
		}

		if ( isset( $request['billing']['company'] ) ) {
			$subscription->set_billing_company( $request['billing']['company'] );
		}

		if ( isset( $request['billing']['address_1'] ) ) {
			$subscription->set_billing_address_1( $request['billing']['address_1'] );
		}

		if ( isset( $request['billing']['address_2'] ) ) {
			$subscription->set_billing_address_2( $request['billing']['address_2'] );
		}

		if ( isset( $request['billing']['city'] ) ) {
			$subscription->set_billing_city( $request['billing']['city'] );
		}

		if ( isset( $request['billing']['postcode'] ) ) {
			$subscription->set_billing_postcode( $request['billing']['postcode'] );
		}

		if ( isset( $request['billing']['country'] ) ) {
			$subscription->set_billing_country( $request['billing']['country'] );
		}

		if ( isset( $request['billing']['state'] ) ) {
			$subscription->set_billing_state( $request['billing']['state'] );
		}

		if ( isset( $request['billing']['email'] ) ) {
			$subscription->set_billing_email( $request['billing']['email'] );
		}

		if ( isset( $request['billing']['phone'] ) ) {
			$subscription->set_billing_phone( $request['billing']['phone'] );
		}

		// Allow set meta_data.
		if ( isset( $request['meta_data'] ) && is_array( $request['meta_data'] ) ) {
			foreach ( $request['meta_data'] as $meta ) {
				$subscription->update_meta_data( $meta['key'], $meta['value'], isset( $meta['id'] ) ? $meta['id'] : '' );
			}
		}

		if ( isset( $request['billing_expire_after'] ) ) {
			$subscription->set_billing_expire_after( $request['billing_expire_after'] );
		}

		if ( isset( $request['billing_expire_after'] ) ) {
			$subscription->set_billing_expire_after( $request['billing_expire_after'] );
		}

		if ( isset( $request['billing_interval'] ) ) {
			$subscription->set_billing_interval( $request['billing_interval'] );
		}

		if ( isset( $request['billing_period'] ) ) {
			$subscription->set_billing_period( $request['billing_period'] );
		}

		if ( isset( $request['payment_method'] ) ) {
			$subscription->set_payment_method( $request['payment_method'] );
		}

		if ( isset( $request['recurring_amount'] ) ) {
			$subscription->set_recurring_amount( $request['recurring_amount'] );
		}

		/**
		 * Filters an object before it is inserted via the REST API.
		 *
		 * The dynamic portion of the hook name, `$this->object_type`,
		 * refers to the object type slug.
		 *
		 * @since 2.6.10
		 *
		 * @param Masteriyo\Database\Model $subscription Subscription object.
		 * @param WP_REST_Request $request  Request object.
		 * @param bool            $creating If is creating a new object.
		 */
		return apply_filters( "masteriyo_rest_pre_insert_{$this->object_type}_object", $subscription, $request, $creating );
	}

	/**
	 * Checks if a given request has access to get a specific item.
	 *
	 * @since 2.6.10
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 * @return boolean|WP_Error True if the request has read access for the item, WP_Error object otherwise.
	 */
	public function get_item_permissions_check( $request ) {
		if ( is_null( $this->permission ) ) {
			return new \WP_Error(
				'masteriyo_null_permission',
				__( 'Sorry, the permission object for this resource is null.', 'learning-management-system' )
			);
		}

		if ( masteriyo_is_current_user_admin() || masteriyo_is_current_user_manager() ) {
			return true;
		}

		$id           = absint( $request['id'] );
		$subscription = masteriyo_get_subscription( $id );

		if ( is_null( $subscription ) ) {
			return new \WP_Error(
				"masteriyo_rest_{$this->post_type}_invalid_id",
				__( 'Invalid ID', 'learning-management-system' ),
				array(
					'status' => 404,
				)
			);
		}

		$cap = $subscription->get_customer_id() === get_current_user_id() ? 'read_subscriptions' : 'read_others_subscriptions';

		if ( ! $this->permission->rest_check_subscription_permissions( $cap, $id ) ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_read',
				__( 'Sorry, you are not allowed to read resources.', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		return true;
	}

	/**
	 * Check if a given request has access to read items.
	 *
	 * @since 2.6.10
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function get_items_permissions_check( $request ) {
		if ( is_null( $this->permission ) ) {
			return new \WP_Error(
				'masteriyo_null_permission',
				__( 'Sorry, the permission object for this resource is null.', 'learning-management-system' )
			);
		}

		if ( masteriyo_is_current_user_admin() || masteriyo_is_current_user_manager() ) {
			return true;
		}

		if ( ! $this->permission->rest_check_subscription_permissions( 'read_subscriptions' ) ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_read',
				__( 'Sorry, you are not allowed to read resources.', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		return true;
	}

	/**
	 * Check permissions for an item.
	 *
	 * @since 2.6.10
	 *
	 * @param string $object_type Object type.
	 * @param string $context   Request context.
	 * @param int    $object_id Post ID.
	 *
	 * @return bool
	 */
	protected function check_item_permission( $object_type, $context = 'read', $object_id = 0 ) {
		if ( masteriyo_is_current_user_admin() || masteriyo_is_current_user_manager() ) {
			return true;
		}

		if ( 'read' === $context ) {
			$subscription = masteriyo_get_subscription( $object_id );
			$cap          = $subscription->get_customer_id() === get_current_user_id() ? 'read_subscriptions' : 'read_others_subscriptions';

			if ( ! $this->permission->rest_check_subscription_permissions( $cap, $object_id ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Check if a given request has access to create an item.
	 *
	 * @since 2.6.10
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function create_item_permissions_check( $request ) {
		if ( is_null( $this->permission ) ) {
			return new \WP_Error(
				'masteriyo_null_permission',
				__( 'Sorry, the permission object for this resource is null.', 'learning-management-system' )
			);
		}

		if ( masteriyo_is_current_user_admin() || masteriyo_is_current_user_manager() ) {
			return true;
		}

		if ( ! $this->permission->rest_check_subscription_permissions( 'create' ) ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_create',
				__( 'Sorry, you are not allowed to create resources.', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		return true;
	}

	/**
	 * Check if a given request has access to delete an item.
	 *
	 * @since 2.6.10
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function delete_item_permissions_check( $request ) {
		if ( is_null( $this->permission ) ) {
			return new \WP_Error(
				'masteriyo_null_permission',
				__( 'Sorry, the permission object for this resource is null.', 'learning-management-system' )
			);
		}

		if ( masteriyo_is_current_user_admin() || masteriyo_is_current_user_manager() ) {
			return true;
		}

		if ( ! $this->permission->rest_check_subscription_permissions( 'delete', $request['id'] ) ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_delete',
				__( 'Sorry, you are not allowed to delete resources.', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		return true;
	}

	/**
	 * Check if a given request has access to update an item.
	 *
	 * @since 2.6.10
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function update_item_permissions_check( $request ) {
		if ( is_null( $this->permission ) ) {
			return new \WP_Error(
				'masteriyo_null_permission',
				__( 'Sorry, the permission object for this resource is null.', 'learning-management-system' )
			);
		}

		if ( masteriyo_is_current_user_admin() || masteriyo_is_current_user_manager() ) {
			return true;
		}

		/**
		 * Prevent from updating the subscription owner to someone else.
		 */
		if ( isset( $request['customer_id'] ) && absint( $request['customer_id'] ) !== get_current_user_id() ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_update',
				__( 'Sorry, you are not allowed to change the owner of the subscription.', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		if ( ! $this->permission->rest_check_subscription_permissions( 'update', $request['id'] ) ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_update',
				__( 'Sorry, you are not allowed to update resources.', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		return true;
	}

	/**
	 * Save an object data.
	 *
	 * @since  2.6.10
	 * @throws RestException But all errors are validated before returning any data.
	 * @param  WP_REST_Request $request  Full details about the request.
	 * @param  bool            $creating If is creating a new object.
	 * @return MOdel|WP_Error
	 */
	protected function save_object( $request, $creating = false ) {
		try {
			$object = $this->prepare_object_for_database( $request, $creating );

			if ( is_wp_error( $object ) ) {
				return $object;
			}

			if ( ! is_null( $request['customer_id'] ) && 0 !== $request['customer_id'] ) {
				// Make sure customer exists.
				if ( false === get_user_by( 'id', $request['customer_id'] ) ) {
					throw new RestException( 'masteriyo_rest_invalid_customer_id', __( 'Customer ID is invalid.', 'learning-management-system' ), 400 );
				}

				// Make sure customer is part of blog.
				if ( is_multisite() && ! is_user_member_of_blog( $request['customer_id'] ) ) {
					add_user_to_blog( get_current_blog_id(), $request['customer_id'], 'masteriyo_student' );
				}
			}

			if ( $creating ) {
				if ( empty( $object->get_created_via() ) ) {
					$object->set_created_via( 'rest-api' );
				}

				$object->set_prices_include_tax( 'yes' === get_option( 'masteriyo_prices_include_tax' ) );
				$object->calculate_totals();
			} else { // phpcs:ignore Universal.ControlStructures.DisallowLonelyIf.Found
				// If items have changed, recalculate subscription totals.
				if ( isset( $request['billing'] ) || isset( $request['course_lines'] ) ) {
					$object->calculate_totals( true );
				}
			}

			// Set status.
			if ( ! empty( $request['status'] ) ) {
				$object->set_status( $request['status'] );
			}

			$object->save();

			// Actions for after the subscription is saved.
			if ( true === $request['set_paid'] ) {
				if ( $creating || $object->needs_payment() ) {
					$object->payment_complete( $request['transaction_id'] );
				}
			}

			return $this->get_object( $object->get_id() );
		} catch ( ModelException $e ) {
			return new \WP_Error( $e->getErrorCode(), $e->getMessage(), $e->getErrorData() );
		} catch ( RestException $e ) {
			return new \WP_Error( $e->getErrorCode(), $e->getMessage(), array( 'status' => $e->getCode() ) );
		}
	}

	/**
	 * Wrapper method to create/update subscription items.
	 * When updating, the item ID provided is checked to ensure it is associated
	 * with the subscription.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Models\Subscription\Subscription $subscription subscription object.
	 * @param string   $item_type The item type.
	 * @param array    $posted item provided in the request body.
	 * @throws RestException If item ID is not associated with subscription.
	 */
	protected function set_item( $subscription, $item_type, $posted ) {
		global $wpdb;

		$action = empty( $posted['id'] ) ? 'create' : 'update';
		$method = 'prepare_' . $item_type;
		$item   = null;

		// Verify provided line item ID is associated with subscription.
		if ( 'update' === $action ) {
			$item = $subscription->get_item( absint( $posted['id'] ), false );

			if ( ! $item ) {
				throw new RestException( 'masteriyo_rest_invalid_item_id', __( 'Subscription item ID provided is not associated with subscription.', 'learning-management-system' ), 400 );
			}
		}

		// Prepare item data.
		$item = $this->$method( $posted, $action, $item );

		/**
		 * Fires before setting an subscription item in an subscription object.
		 *
		 * @since 2.6.10
		 *
		 * @param mixed $item Subscription item.
		 * @param mixed $posted Posted data from request.
		 */
		do_action( 'masteriyo_rest_set_subscription_item', $item, $posted );

		// If creating the subscription, add the item to it.
		if ( 'create' === $action ) {
			$subscription->add_item( $item );
		} else {
			$item->save();
		}
	}


	/**
	 * Gets the course ID from the SKU or posted ID.
	 *
	 * @since 2.6.10
	 *
	 * @throws RestException When SKU or ID is not valid.
	 * @param array  $posted Request data.
	 * @param string $action 'create' to add line item or 'update' to update it.
	 * @return int
	 */
	protected function get_course_id( $posted, $action = 'create' ) {
		if ( ! empty( $posted['course_id'] ) && empty( $posted['variation_id'] ) ) {
			$course_id = (int) $posted['course_id'];
		} elseif ( 'update' === $action ) {
			$course_id = 0;
		} else {
			throw new RestException( 'masteriyo_rest_required_course_reference', __( 'Course ID or SKU is required.', 'learning-management-system' ), 400 );
		}
		return $course_id;
	}

	/**
	 * Restore subscription.
	 *
	 * @since 2.6.10
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|WP_REST_Response
	 */
	public function restore_item( $request ) {
		$object = $this->get_object( (int) $request['id'] );

		if ( ! $object || 0 === $object->get_id() ) {
			return new \WP_Error( "masteriyo_rest_{$this->object_type}_invalid_id", __( 'Invalid ID.', 'learning-management-system' ), array( 'status' => 404 ) );
		}

		$object->restore();

		$data     = $this->prepare_object_for_response( $object, $request );
		$response = rest_ensure_response( $data );

		if ( $this->public ) {
			$response->link_header( 'alternate', $this->get_permalink( $object ), array( 'type' => 'text/html' ) );
		}

		return $response;
	}

	/**
	 * Get courses count by status.
	 *
	 * @since 2.6.10
	 *
	 * @return array
	 */
	protected function get_subscriptions_count() {
		$post_count = parent::get_posts_count();

		$post_count        = masteriyo_array_only( $post_count, SubscriptionStatus::all() );
		$post_count['any'] = array_sum( masteriyo_array_except( $post_count, SubscriptionStatus::TRASH ) );

		return $post_count;
	}

	/**
	 * Get the query params for collections of attachments.
	 *
	 * @since 2.6.10
	 *
	 * @return array
	 */
	public function get_collection_params() {
		$params = parent::get_collection_params();

		$params['status'] = array(
			'description'       => __( ' Limit result set to subscriptions assigned a specific status.', 'learning-management-system' ),
			'type'              => 'string',
			'default'           => 'any',
			'enum'              => array_merge( array_keys( SubscriptionStatus::list() ), array( 'any', 'trash' ) ),
			'sanitize_callback' => 'sanitize_key',
			'validate_callback' => 'rest_validate_request_arg',
		);

		return $params;
	}

	/**
	 * Return collection params for logged in user.
	 *
	 * @since 2.6.10
	 *
	 * @return array
	 */
	public function get_user_collection_params() {
		$params = $this->get_collection_params();

		unset( $params['customer'] );
		unset( $params['include'] );
		unset( $params['exclude'] );

		return $params;
	}

	/**
	 * Return the logged in user subscriptions.
	 *
	 * @since 2.6.10
	 *
	 * @param WP_REST_Request $request Full details about the request.
	 *
	 * @return WP_Error|WP_REST_Response
	 */
	public function get_user_subscriptions( $request ) {
		unset( $request['customer'] );
		unset( $request['include'] );
		unset( $request['exclude'] );

		$request->set_param( 'customer', get_current_user_id() );

		return $this->get_items( $request );
	}

	/**
	 * Get courses count by status.
	 *
	 * @since 2.6.10
	 *
	 * @return array
	 */
	protected function get_orders_count() {
		$post_count = parent::get_posts_count();

		$post_count        = masteriyo_array_only( $post_count, SubscriptionStatus::all() );
		$post_count['any'] = array_sum( masteriyo_array_except( $post_count, SubscriptionStatus::TRASH ) );

		return $post_count;
	}

	/**
	 * Check if a given request has access to delete items.
	 *
	 * @since 2.6.10
	 *
	 * @param  \WP_REST_Request $request Full details about the request.
	 * @return \WP_Error|boolean
	 */
	public function delete_items_permissions_check( $request ) {
		if ( is_null( $this->permission ) ) {
			return new \WP_Error(
				'masteriyo_null_permission',
				__( 'Sorry, the permission object for this resource is null.', 'learning-management-system' )
			);
		}

		if ( masteriyo_is_current_user_admin() || masteriyo_is_current_user_manager() ) {
			return true;
		}

		if ( ! $this->permission->rest_check_post_permissions( $this->post_type, 'batch' ) ) {
			return new \WP_Error(
				'masteriyo_rest_cannot_read',
				__( 'Sorry, you are not allowed to delete resources', 'learning-management-system' ),
				array(
					'status' => rest_authorization_required_code(),
				)
			);
		}

		return true;
	}

	/**
	 * Prepare objects query.
	 *
	 * @param \WP_REST_Request $request Full details about the request.
	 *
	 * @since  2.6.10
	 * @return array
	 */
	protected function prepare_objects_query_for_batch( $request ) {
		$query_args = parent::prepare_objects_query_for_batch( $request );

		$query_args['post_status'] = SubscriptionStatus::all();

		/**
		 * Filters objects query for batch operation.
		 *
		 * @since 1.6.5
		 *
		 * @param array $query_args Query arguments.
		 * @param \WP_REST_Request $request
		 * @param \Masteriyo\RestApi\Controllers\Version1\PostsController $controller
		 */
		return apply_filters( "masteriyo_rest_{$this->object_type}_objects_query_for_batch", $query_args, $request, $this );
	}
}
