<?php
/**
 * Subscription status enums.
 *
 * @since 2.6.10
 * @package Masteriyo\Pro\Enums
 */

namespace Masteriyo\Pro\Enums;

use Masteriyo\Enums\OrderStatus;

defined( 'ABSPATH' ) || exit;

/**
 * Subscription status enum class.
 *
 * @since 2.6.10
 */
class SubscriptionStatus extends OrderStatus {
	/**
	 * Subscription active status.
	 *
	 * @since 2.6.10
	 * @var string
	 */
	const ACTIVE = 'active';

	/**
	 * Subscription switched status.
	 *
	 * @since 2.6.10
	 * @var string
	 */
	const SWITCHED = 'switched';

	/**
	 * Subscription expired status.
	 *
	 * @since 2.6.10
	 * @var string
	 */
	const EXPIRED = 'expired';

	/**
	 * Subscription pending-cancel status.
	 *
	 * @since 2.6.10
	 * @var string
	 */
	const PENDING_CANCEL = 'pending-cancel';

	/**
	 * Return all subscription statuses.
	 *
	 * @since 2.6.10
	 *
	 * @return array
	 */
	public static function all() {
		return array_unique(
			/**
			 * Filters subscription status list.
			 *
			 * @since 2.6.10
			 *
			 * @param string[] $statuses Subscription status list.
			 */
			apply_filters(
				'masteriyo_subscription_statuses',
				array(
					self::PENDING,
					self::ACTIVE,
					self::COMPLETED,
					self::ON_HOLD,
					self::SWITCHED,
					self::EXPIRED,
					self::PENDING_CANCEL,
					self::TRASH,
					self::CANCELLED,
				)
			)
		);
	}

	/**
	 * Return label for the status.
	 *
	 * @since 2.6.10
	 *
	 * @param string $status
	 * @return string|null
	 */
	public static function label( $status ) {
		$labels =
			/**
			 * Filters subscription labels.
			 *
			 * @since 2.6.10
			 *
			 * @param string[] Subscription status labels.
			 */
			apply_filters(
				'masteriyo_subscription_statuses_labels',
				array(
					'pending'        => _x( 'Pending', 'Subscription status', 'learning-management-system' ),
					'active'         => _x( 'Active', 'Subscription status', 'learning-management-system' ),
					'completed'      => _x( 'Completed', 'Subscription status', 'learning-management-system' ),
					'on-hold'        => _x( 'On hold', 'Subscription status', 'learning-management-system' ),
					'cancelled'      => _x( 'Cancelled', 'Subscription status', 'learning-management-system' ),
					'switched'       => _x( 'Switched', 'Subscription status', 'learning-management-system' ),
					'expired'        => _x( 'Expired', 'Subscription status', 'learning-management-system' ),
					'pending-cancel' => _x( 'Pending Cancellation', 'Subscription status', 'learning-management-system' ),
				)
			);

		return isset( $labels[ $status ] ) ? $labels[ $status ] : null;
	}

	/**
	 * Return list of objects mainly used for registering post status.
	 *
	 * @since 2.6.10
	 *
	 * @return array
	 */
	public static function list() {
		$subscription_statuses = array(
			self::PENDING        => array(
				'label'                     => _x( 'Pending payment', 'Subscription status', 'learning-management-system' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				// translators: %s: number of subscriptions
				'label_count'               => _n_noop( 'Pending payment <span class="count">(%s)</span>', 'Pending payment <span class="count">(%s)</span>', 'learning-management-system' ),
			),
			self::ACTIVE         => array(
				'label'                     => _x( 'Active', 'Subscription status', 'learning-management-system' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				// translators: %s: number of subscriptions
				'label_count'               => _n_noop( 'Active <span class="count">(%s)</span>', 'Active payment <span class="count">(%s)</span>', 'learning-management-system' ),
			),
			self::COMPLETED      => array(
				'label'                     => _x( 'Completed', 'Subscription status', 'learning-management-system' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				// translators: %s: number of subscriptions
				'label_count'               => _n_noop( 'Active <span class="count">(%s)</span>', 'Completed payment <span class="count">(%s)</span>', 'learning-management-system' ),
			),
			self::ON_HOLD        => array(
				'label'                     => _x( 'On hold', 'Subscription status', 'learning-management-system' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				/* translators: %s: number of subscriptions */
				'label_count'               => _n_noop( 'On hold <span class="count">(%s)</span>', 'On hold <span class="count">(%s)</span>', 'learning-management-system' ),
			),
			self::SWITCHED       => array(
				'label'                     => _x( 'Switched', 'Subscription status', 'learning-management-system' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				/* translators: %s: number of subscriptions */
				'label_count'               => _n_noop( 'Switched <span class="count">(%s)</span>', 'Switched <span class="count">(%s)</span>', 'learning-management-system' ),
			),
			self::EXPIRED        => array(
				'label'                     => _x( 'Expired', 'Subscription status', 'learning-management-system' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				/* translators: %s: number of subscriptions */
				'label_count'               => _n_noop( 'Expired <span class="count">(%s)</span>', 'Cancelled <span class="count">(%s)</span>', 'learning-management-system' ),
			),
			self::PENDING_CANCEL => array(
				'label'                     => _x( 'Pending cancel', 'Subscription status', 'learning-management-system' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				/* translators: %s: number of subscriptions */
				'label_count'               => _n_noop( 'Pending cancel <span class="count">(%s)</span>', 'Pending cancel <span class="count">(%s)</span>', 'learning-management-system' ),
			),
			self::CANCELLED      => array(
				'label'                     => _x( 'Cancelled', 'Subscription status', 'learning-management-system' ),
				'public'                    => true,
				'exclude_from_search'       => false,
				'show_in_admin_all_list'    => true,
				'show_in_admin_status_list' => true,
				/* translators: %s: number of subscriptions */
				'label_count'               => _n_noop( 'Cancelled <span class="count">(%s)</span>', 'Cancelled <span class="count">(%s)</span>', 'learning-management-system' ),
			),
		);

		/**
		 * Filters subscription statuses.
		 *
		 * @since 2.6.10
		 *
		 * @param array $subscription_statuses The subscription statuses and its parameters.
		 */
		return apply_filters( 'masteriyo_subscription_statuses_list', $subscription_statuses );
	}
}
