<?php
/**
 * Pro utility functions.
 *
 * @since 2.5.0
 */

/**
 * Return variant name from id.
 *
 * @since 2.5.0
 *
 * @param int $variant_id Variant ID.
 * @return int
 */
function masteriyo_pro_get_variant_name_from_id( $variant_id ) {
	$variant_id = strval( $variant_id );

	$map = array(
		'8795' => 'Starter',
		'8796' => 'Growth',
		'8797' => 'Scale',
	);

	$variant_name = isset( $map[ $variant_id ] ) ? $map[ $variant_id ] : 'Unknown';

	return $variant_name;
}


/**
 * Return variant ID from name.
 *
 * @since 2.5.0
 *
 * @param string $variant_name Variant name.
 * @return int|null
 */
function masteriyo_pro_get_variant_id_from_name( $variant_name ) {
	$variant_name = masteriyo_strtolower( $variant_name );

	$map = array(
		'starter' => 8795,
		'growth'  => 8796,
		'scale'   => 8797,
	);

	$variant_id = isset( $map[ $variant_name ] ) ? $map[ $variant_name ] : null;

	return $variant_id;
}


/**
 * Return instance name.
 *
 * @since 2.5.0
 *
 * @return string
 */
function masteriyo_pro_get_instance_name() {
	$instance_name = wp_parse_url( get_bloginfo( 'url' ), PHP_URL_HOST );
	$instance_name = $instance_name ? $instance_name : wp_generate_uuid4();

	return $instance_name;
}

/**
 * Activate license key.
 *
 * @since 2.5.0
 *
 * @param string $license_key License key.
 * @param string $instance_name Instance name.
 * @return array|WP_Error
 */
function masteriyo_pro_activate_license( $license_key, $instance_name ) {
	$response = wp_remote_get(
		MASTERIYO_PRO_API_URL . '/activate',
		array(
			'headers' => array(
				'Accept' => 'application/json',
			),
			'timeout' => 30,
			'body'    => array(
				'license_key'   => $license_key,
				'instance_name' => $instance_name,
			),
		)
	);

	if ( is_wp_error( $response ) ) {
		return $response;
	}

	// Throw error when firewall flags license activation process as malicious activity.
	if ( 403 === wp_remote_retrieve_response_code( $response ) ) {
		return new \WP_Error(
			'masteriyo_pro_rest_license_authorized',
			wp_remote_retrieve_body( $response ),
			array( 'status' => 403 )
		);
	}

	$data = json_decode( wp_remote_retrieve_body( $response ), true );

	if ( is_null( $data ) ) {
		return new \WP_Error(
			'masteriyo_pro_rest_invalid_json',
			esc_html__( 'Invalid JSON received from the validation server.', 'learning-management-system' ),
			array( 'status' => 500 )
		);
	}

	$activated = masteriyo_array_get( $data, 'data.activated', false );

	if ( false === $activated ) {
		return new \WP_Error(
			'masteriyo_pro_rest_license_activation_fail',
			$data['error'],
			array(
				'status' => 400,
			)
		);
	}

	return masteriyo_array_get( $data, 'data' );
}

/**
 * Deactivate license key.
 *
 * @since 2.5.0
 *
 * @param string $license_key
 * @param string $instance_id
 * @return boolean|WP_Error
 */
function masteriyo_pro_deactivate_license( $license_key, $instance_id ) {
	$deactivate_url = add_query_arg(
		array(
			'license_key' => $license_key,
			'instance_id' => $instance_id,
		),
		MASTERIYO_PRO_API_URL . '/deactivate'
	);

	$response = wp_remote_get(
		$deactivate_url,
		array(
			'headers' => array(
				'Accept' => 'application/json',
			),
			'timeout' => 30,
		)
	);

	if ( is_wp_error( $response ) ) {
		return $response;
	}

	// Throw error when firewall flags license deactivation process as malicious activity.
	if ( 403 === wp_remote_retrieve_response_code( $response ) ) {
		return new \WP_Error(
			'masteriyo_pro_rest_license_authorized',
			wp_remote_retrieve_body( $response ),
			array( 'status' => 403 )
		);
	}

	$data = json_decode( wp_remote_retrieve_body( $response ), true );

	if ( is_null( $data ) ) {
		return new \WP_Error(
			'masteriyo_pro_rest_invalid_json',
			esc_html__( 'Invalid JSON received from the validation server.', 'learning-management-system' ),
			array( 'status' => 500 )
		);
	}

	return isset( $data['success'] );
}

/**
 * Validate the license key.
 *
 * @since 2.6.7
 *
 * @param string $license_key The key of the license.
 *
 * @return boolean Returns false if only the license key is invalid or expired and return true otherwise.
 */
function masteriyo_pro_validate_license( $license_key ) {
	$validate_url = add_query_arg(
		array(
			'license_key' => $license_key,
		),
		MASTERIYO_PRO_API_URL . '/update'
	);

	$response = wp_remote_get(
		$validate_url,
		array(
			'headers' => array(
				'Accept' => 'application/json',
			),
			'timeout' => 10,
		)
	);

	if ( is_wp_error( $response ) ) {
		return true;
	}

	$data = json_decode( wp_remote_retrieve_body( $response ), true );

	if ( is_array( $data ) && isset( $data['error_code'] ) && 'invalid_license_key' === $data['error_code'] ) {
		return false;
	}

	return true;
}
