<?php
/**
 * Subscription model service provider.
 *
 * @since 2.6.10
 */

namespace Masteriyo\Pro\Providers;

use Masteriyo\DateTime;

defined( 'ABSPATH' ) || exit;

use Masteriyo\Enums\CommentType;
use Masteriyo\PostType\PostType;
use Masteriyo\Pro\Models\Subscription;
use Masteriyo\Query\SubscriptionQuery;
use Masteriyo\Pro\Enums\SubscriptionStatus;
use Masteriyo\Pro\Repository\SubscriptionRepository;
use Masteriyo\Pro\Controllers\SubscriptionsController;
use League\Container\ServiceProvider\AbstractServiceProvider;
use League\Container\ServiceProvider\BootableServiceProviderInterface;
use Masteriyo\Pro\Jobs\CheckSubscriptionExpirationJob;

class SubscriptionServiceProvider extends AbstractServiceProvider implements BootableServiceProviderInterface {
	/**
	 * The provided array is a way to let the container
	 * know that a service is provided by this service
	 * provider. Every service that is registered via
	 * this service provider must have an alias added
	 * to this array or it will be ignored
	 *
	 * @since 2.6.10
	 *
	 * @var array
	 */
	protected $provides = array(
		'subscription',
		'subscription.store',
		'subscription.rest',
		'mto-subscription',
		'mto-subscription.store',
		'mto-subscription.rest',
		SubscriptionsController::class,
	);

	/**
	 * This is where the magic happens, within the method you can
	* access the container and register or retrieve anything
	* that you need to, but remember, every alias registered
	* within this method must be declared in the `$provides` array.
	*
	* @since 2.6.10
	*/
	public function register() {
		$this->getContainer()->add( SubscriptionsController::class )->addArgument( 'permission' );

		$this->getContainer()->add( 'subscription.store', SubscriptionRepository::class );

		$this->getContainer()->add( 'subscription.rest', SubscriptionsController::class )
			->addArgument( 'permission' );

		$this->getContainer()->add( 'subscription', Subscription::class )
			->addArgument( 'subscription.store' );

		$this->getContainer()->add( 'mto-subscription.store', SubscriptionRepository::class );

		$this->getContainer()->add( 'mto-subscription.rest', SubscriptionsController::class )
			->addArgument( 'permission' );

		$this->getContainer()->add( 'subscription', Subscription::class )
			->addArgument( 'mto-subscription.store' );
	}

	/**
	 * In much the same way, this method has access to the container
	 * itself and can interact with it however you wish, the difference
	 * is that the boot method is invoked as soon as you register
	 * the service provider with the container meaning that everything
	 * in this method is eagerly loaded.
	 *
	 * If you wish to apply inflectors or register further service providers
	 * from this one, it must be from a bootable service provider like
	 * this one, otherwise they will be ignored.
	 *
	 * @since 2.6.10
	 */
	public function boot() {
		add_filter( 'comments_open', array( $this, 'comments_open' ), 10, 2 );
		add_action( 'comment_moderation_recipients', array( $this, 'comment_moderation_recipients' ), 10, 2 );
		add_action( 'parse_comment_query', array( $this, 'remove_subscription_note_from_query' ) );
		add_action( 'masteriyo_checkout_before_order_summary', array( $this, 'render_subscription_summary_in_checkout' ) );
		$this->register_subscription_expiration_job();
	}

	public function render_subscription_summary_in_checkout( $order ) {
		if ( masteriyo_order_contains_recurring_courses( $order ) ) {
			$subscription = masteriyo_get_subscription_from_order( $order );
			if ( $subscription ) {
				masteriyo_get_template( 'checkout/subscription.php', array( 'subscription' => $subscription ) );
			}
		}
	}

	/**
	 * See if comments are open.
	 *
	 * @since 2.6.10
	 *
	 * @param  bool $open    Whether the current post is open for comments.
	 * @param  int  $post_id Post ID.
	 *
	 * @return bool
	 */
	public function comments_open( $open, $post_id ) {
		if ( PostType::SUBSCRIPTION === get_post_type( $post_id ) ) {
			$open = false;
		}
		return $open;
	}

	/**
	 * Modify recipient of review email.
	 *
	 * @since 2.6.10
	 *
	 * @param array $emails     Emails.
	 * @param int   $comment_id Comment ID.
	 *
	 * @return array
	 */
	public function comment_moderation_recipients( $emails, $comment_id ) {
		$comment = get_comment( $comment_id );

		if ( $comment && PostType::SUBSCRIPTION === get_post_type( $comment->comment_post_ID ) ) {
			$emails = array( get_option( 'admin_email' ) );
		}

		return $emails;
	}

	/**
	 * Remove the course review from the comments query.
	 *
	 * @since 2.6.10
	 *
	 * @param \WP_Comment_Query $query
	 */
	public function remove_subscription_note_from_query( $query ) {
		// Bail early if  global pagenow is not set or isn't admin dashboard.
		if ( ! isset( $GLOBALS['pagenow'] ) || ! is_admin() ) {
			return;
		}

		// Bail if the page is not wp comments list page or dashboard.
		if ( ! in_array( $GLOBALS['pagenow'], array( 'edit-comments.php', 'index.php' ), true ) ) {
			return;
		}

		if ( ! isset( $query->query_vars['type__not_in'] ) ) {
			$query->query_vars['type__not_in'] = array();
		}

		$query->query_vars['type__not_in'] = (array) $query->query_vars['type__not_in'];
		$query->query_vars['type__not_in'] = array_unique( array_merge( $query->query_vars['type__not_in'], array( CommentType::SUBSCRIPTION_NOTE ) ) );
	}

	/**
	 * Register subscription expiration job.
	 *
	 * @since 2.6.10
	 */
	public function register_subscription_expiration_job() {
		add_action(
			'init',
			function() {

				$next = as_next_scheduled_action( CheckSubscriptionExpirationJob::HOOK, array(), 'learning-management-system' );

				if ( false === $next ) {
					as_schedule_recurring_action( time(), DAY_IN_SECONDS, CheckSubscriptionExpirationJob::HOOK, array(), 'learning-management-system' );
				}
			}
		);
		( new CheckSubscriptionExpirationJob() )->init();
	}
}
