<?php

/**
 * Subscription repository class.
 *
 * @since 2.6.10
 *
 * @package Masteriyo\Repository
 */

namespace Masteriyo\Pro\Repository;

defined( 'ABSPATH' ) || exit;

use Masteriyo\Constants;
use Masteriyo\Database\Model;
use Masteriyo\PostType\PostType;
use Masteriyo\Enums\SubscriptionStatus;
use Masteriyo\Repository\OrderRepository;
use Masteriyo\Repository\RepositoryInterface;

/**
 * SubscriptionRepository class.
 */
class SubscriptionRepository extends OrderRepository implements RepositoryInterface {

	/**
	 * Create a subscription in the database.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Pro\Models\Subscription $subscription subscription object.
	 */
	public function create( Model &$subscription ) {
		if ( ! $subscription->get_date_created( 'edit' ) ) {
			$subscription->set_date_created( time() );
		}

		if ( '' === $subscription->get_order_key() ) {
			$subscription->set_order_key( masteriyo_generate_order_key() );
		}

		$subscription->set_currency( $subscription->get_currency() ? $subscription->get_currency() : masteriyo_get_currency() );
		$subscription->set_version( Constants::get( 'MASTERIYO_VERSION' ) );

		$id = wp_insert_post(
			/**
			 * Filters new subscription data before creating.
			 *
			 * @since 2.6.10
			 *
			 * @param array $data New subscription data.
			 * @param \Masteriyo\Pro\Models\Subscription $subscription Subscription object.
			 */
			apply_filters(
				'masteriyo_new_subscription_data',
				array(
					'post_type'     => PostType::SUBSCRIPTION,
					'post_status'   => $subscription->get_status() ? $subscription->get_status() : SubscriptionStatus::PENDING,
					'post_author'   => 1,
					'post_title'    => $this->get_subscription_title(),
					'post_password' => $this->get_order_key( $subscription ),
					'ping_status'   => 'closed',
					'post_excerpt'  => $subscription->get_customer_note( 'edit' ),
					'post_date'     => gmdate( 'Y-m-d H:i:s', $subscription->get_date_created( 'edit' )->getOffsetTimestamp() ),
					'post_date_gmt' => gmdate( 'Y-m-d H:i:s', $subscription->get_date_created( 'edit' )->getTimestamp() ),
					'post_parent'   => $subscription->get_parent_id( 'edit' ),
				),
				$subscription
			),
			true
		);

		if ( $id && ! is_wp_error( $id ) ) {
			$subscription->set_id( $id );
			$this->update_post_meta( $subscription, true );
			// TODO Invalidate caches.

			$subscription->save_meta_data();
			$subscription->apply_changes();
			$this->clear_caches( $subscription );

			/**
			 * Fires after creating an subscription.
			 *
			 * @since 2.6.10
			 *
			 * @param integer $id The subscription ID.
			 * @param \Masteriyo\Pro\Models\Subscription $object The subscription object.
			 */
			do_action( 'masteriyo_new_subscription', $id, $subscription );
		}
	}

	/**
	 * Read an subscription.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Pro\Models\Subscription $subscription Subscription object.
	 *
	 * @throws \Exception If invalid subscription.
	 */
	public function read( Model &$subscription ) {
		$subscription_post = get_post( $subscription->get_id() );

		if ( ! $subscription->get_id() || ! $subscription_post || PostType::SUBSCRIPTION !== $subscription_post->post_type ) {
			throw new \Exception( __( 'Invalid subscription.', 'learning-management-system' ) );
		}

		$subscription->set_props(
			array(
				'status'        => $subscription_post->post_status,
				'date_created'  => $this->string_to_timestamp( $subscription_post->post_date_gmt ),
				'date_modified' => $this->string_to_timestamp( $subscription_post->post_modified_gmt ),
				'customer_note' => $subscription_post->post_excerpt,
				'parent_id'     => $subscription_post->post_parent,
			)
		);

		$this->read_subscription_data( $subscription );
		$this->read_extra_data( $subscription );
		$subscription->set_object_read( true );

		/**
		 * Fires after reading an subscription object from database.
		 *
		 * @since 2.6.10
		 *
		 * @param integer $id The subscription ID.
		 * @param \Masteriyo\Pro\Models\Subscription $object The subscription object.
		 */
		do_action( 'masteriyo_subscription_read', $subscription->get_id(), $subscription );
	}

	/**
	 * Update an subscription in the database.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Pro\Models\Subscription $subscription subscription object.
	 */
	public function update( Model &$subscription ) {
		$subscription->set_version( Constants::get( 'MASTERIYO_VERSION' ) );

		$changes = $subscription->get_changes();

		$post_data_keys = array(
			'status',
			'parent_id',
			'date_modified',
		);

		// Only update the post when the post data changes.
		if ( array_intersect( $post_data_keys, array_keys( $changes ) ) ) {
			$post_data = array(
				'post_status' => $subscription->get_status( 'edit' ),
				'post_type'   => PostType::SUBSCRIPTION,
				'post_parent' => $subscription->get_parent_id(),
			);

			/**
			 * When updating this object, to prevent infinite loops, use $wpdb
			 * to update data, since wp_update_post spawns more calls to the
			 * save_post action.
			 *
			 * This ensures hooks are fired by either WP itself (admin screen save),
			 * or an update purely from CRUD.
			 */
			if ( doing_action( 'save_post' ) ) {
				// TODO Abstract the $wpdb WordPress class.
				$GLOBALS['wpdb']->update( $GLOBALS['wpdb']->posts, $post_data, array( 'ID' => $subscription->get_id() ) );
				clean_post_cache( $subscription->get_id() );
			} else {
				wp_update_post( array_merge( array( 'ID' => $subscription->get_id() ), $post_data ) );
			}
			$subscription->read_meta_data( true ); // Refresh internal meta data, in case things were hooked into `save_post` or another WP hook.
		} else { // Only update post modified time to record this save event.
			$GLOBALS['wpdb']->update(
				$GLOBALS['wpdb']->posts,
				array(
					'post_modified'     => current_time( 'mysql' ),
					'post_modified_gmt' => current_time( 'mysql', true ),
				),
				array(
					'ID' => $subscription->get_id(),
				)
			);
			clean_post_cache( $subscription->get_id() );
		}

		$this->update_post_meta( $subscription );

		$subscription->apply_changes();

		/**
		 * Fires after updating an subscription object in database.
		 *
		 * @since 2.6.10
		 *
		 * @param integer $id The subscription ID.
		 * @param \Masteriyo\Pro\Models\Subscription $object The subscription object.
		 */
		do_action( 'masteriyo_update_subscription', $subscription->get_id(), $subscription );
	}

	/**
	 * Delete an subscription from the database.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Pro\Models\Subscription $subscription subscription object.
	 * @param array $args   Array of args to pass.
	 */
	public function delete( Model &$subscription, $args = array() ) {
		$id          = $subscription->get_id();
		$object_type = $subscription->get_object_type();

		$args = array_merge(
			array(
				'force_delete' => false,
			),
			$args
		);

		if ( ! $id ) {
			return;
		}

		if ( $args['force_delete'] ) {
			/**
			 * Fires before deleting an subscription object from database.
			 *
			 * @since 2.6.10
			 *
			 * @param integer $id The subscription ID.
			 * @param \Masteriyo\Pro\Models\Subscription $object The subscription object.
			 */
			do_action( 'masteriyo_before_delete_' . $object_type, $id, $subscription );

			wp_delete_post( $id, true );
			$subscription->set_id( 0 );

			/**
			 * Fires after deleting an subscription object from database.
			 *
			 * @since 2.6.10
			 *
			 * @param integer $id The subscription ID.
			 * @param \Masteriyo\Pro\Models\Subscription $object The subscription object.
			 */
			do_action( 'masteriyo_after_delete_' . $object_type, $id, $subscription );
		} else {
			/**
			 * Fires before moving an subscription to trash.
			 *
			 * @since 2.6.10
			 *
			 * @param integer $id The subscription ID.
			 * @param \Masteriyo\Pro\Models\Subscription $object The subscription object.
			 */
			do_action( 'masteriyo_before_trash_' . $object_type, $id, $subscription );

			wp_trash_post( $id );
			$subscription->set_status( 'trash' );

			/**
			 * Fires after moving an subscription to trash.
			 *
			 * @since 2.6.10
			 *
			 * @param integer $id The subscription ID.
			 * @param \Masteriyo\Pro\Models\Subscription $object The subscription object.
			 */
			do_action( 'masteriyo_after_trash_' . $object_type, $id, $subscription );
		}
	}

	/**
	 * Restore an subscription from the database to previous status.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Pro\Models\Subscription $subscription Subscription object.
	 * @param array $args   Array of args to pass.
	 */
	public function restore( &$subscription, $args = array() ) {

		$previous_status = get_post_meta( $subscription->get_id(), '_wp_trash_meta_status', true );

		wp_untrash_post( $subscription->get_id() );

		$subscription->set_status( $previous_status );

		$post_data = array(
			'post_status'       => $subscription->get_status( 'edit' ),
			'post_type'         => PostType::SUBSCRIPTION,
			'post_modified'     => current_time( 'mysql' ),
			'post_modified_gmt' => current_time( 'mysql', true ),
		);

		/**
		 * When updating this object, to prevent infinite loops, use $wpdb
		 * to update data, since wp_update_post spawns more calls to the
		 * save_post action.
		 *
		 * This ensures hooks are fired by either WP itself (admin screen save),
		 * or an update purely from CRUD.
		 */
		if ( doing_action( 'save_post' ) ) {
			// TODO Abstract the $wpdb WordPress class.
			$GLOBALS['wpdb']->update( $GLOBALS['wpdb']->posts, $post_data, array( 'ID' => $subscription->get_id() ) );
		} else {
			wp_update_post( array_merge( array( 'ID' => $subscription->get_id() ), $post_data ) );
		}
		clean_post_cache( $subscription->get_id() );
	}

	/**
	 * Read subscription data. Can be overridden by child classes to load other props.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Pro\Models\Subscription $subscription subscription object.
	 */
	protected function read_subscription_data( &$subscription ) {
		$id          = $subscription->get_id();
		$meta_values = $this->read_meta( $subscription );

		$set_props = array();

		$meta_values = array_reduce(
			$meta_values,
			function ( $result, $meta_value ) {
				$result[ $meta_value->key ][] = $meta_value->value;
				return $result;
			},
			array()
		);

		foreach ( $this->internal_meta_keys as $prop => $meta_key ) {
			$meta_value         = isset( $meta_values[ $meta_key ][0] ) ? $meta_values[ $meta_key ][0] : null;
			$set_props[ $prop ] = maybe_unserialize( $meta_value ); // get_post_meta only unserializes single values.
		}

		$subscription->set_props( $set_props );
	}

	/**
	 * Read extra data associated with the subscription.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Pro\Models\Subscription $subscription Subscription object.
	 */
	protected function read_extra_data( &$subscription ) {
		$meta_values = $this->read_meta( $subscription );

		foreach ( $subscription->get_extra_data_keys() as $key ) {
			$function = 'set_' . $key;
			if ( is_callable( array( $subscription, $function ) )
				&& isset( $meta_values[ '_' . $key ] ) ) {
				$subscription->{$function}( $meta_values[ '_' . $key ] );
			}
		}
	}

	/**
	 * Fetch subscriptions.
	 *
	 * @since 2.6.10
	 *
	 * @param array $query_vars Query vars.
	 * @return \Masteriyo\Pro\Models\Subscription[]
	 */
	public function query( $query_vars ) {
		$args = $this->get_wp_query_args( $query_vars );

		if ( ! empty( $args['errors'] ) ) {
			$query = (object) array(
				'posts'         => array(),
				'found_posts'   => 0,
				'max_num_pages' => 0,
			);
		} else {
			$query = new \WP_Query( $args );
		}

		if ( isset( $query_vars['return'] ) && 'objects' === $query_vars['return'] && ! empty( $query->posts ) ) {
			// Prime caches before grabbing objects.
			update_post_caches( $query->posts, array( PostType::SUBSCRIPTION ) );
		}

		$subscriptions = ( isset( $query_vars['return'] ) && 'ids' === $query_vars['return'] ) ? $query->posts : array_filter( array_map( 'masteriyo_get_subscription', $query->posts ) );

		if ( isset( $query_vars['paginate'] ) && $query_vars['paginate'] ) {
			return (object) array(
				'subscriptions' => $subscriptions,
				'total'         => $query->found_posts,
				'max_num_pages' => $query->max_num_pages,
			);
		}

		return $subscriptions;
	}

	/**
	 * Get valid WP_Query args from a Subscription's query variables.
	 *
	 * @since 2.6.10
	 * @param array $query_vars Query vars from a Subscription.
	 * @return array
	 */
	protected function get_wp_query_args( $query_vars ) {
		// Map customer id.
		if ( isset( $query_vars['customer_id'] ) ) {
			$query_vars['author'] = $query_vars['customer_id'];
			unset( $query_vars['customer_id'] );
		}

		$wp_query_args = parent::get_wp_query_args( $query_vars );

		if ( ! isset( $wp_query_args['date_query'] ) ) {
			$wp_query_args['date_query'] = array();
		}
		if ( ! isset( $wp_query_args['meta_query'] ) ) {
			$wp_query_args['meta_query'] = array(); // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
		}

		// Handle date queries.
		$date_queries = array(
			'date_created'   => 'post_date',
			'date_modified'  => 'post_modified',
			'date_paid'      => '_date_paid',
			'date_completed' => '_date_completed',
		);
		foreach ( $date_queries as $query_var_key => $db_key ) {
			if ( isset( $query_vars[ $query_var_key ] ) && '' !== $query_vars[ $query_var_key ] ) {

				// Remove any existing meta queries for the same keys to prevent conflicts.
				$existing_queries = wp_list_pluck( $wp_query_args['meta_query'], 'key', true );
				foreach ( $existing_queries as $query_index => $query_contents ) {
					unset( $wp_query_args['meta_query'][ $query_index ] );
				}

				$wp_query_args = $this->parse_date_for_wp_query( $query_vars[ $query_var_key ], $db_key, $wp_query_args );
			}
		}

		// Handle paginate.
		if ( ! isset( $query_vars['paginate'] ) || ! $query_vars['paginate'] ) {
			$wp_query_args['no_found_rows'] = true;
		}

		// Handle orderby.
		if ( isset( $query_vars['orderby'] ) && 'include' === $query_vars['orderby'] ) {
			$wp_query_args['orderby'] = 'post__in';
		}

		/**
		 * Filters WP Query args for subscription post type query.
		 *
		 * @since 2.6.10
		 *
		 * @param array $wp_query_args WP Query args.
		 * @param array $query_vars Query vars.
		 * @param SubscriptionRepository $repository Subscription repository object.
		 */
		return apply_filters( 'masteriyo_subscription_data_store_cpt_get_subscriptions_query', $wp_query_args, $query_vars, $this );
	}

	/**
	 * Clear any caches.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Pro\Models\Subscription $subscription Subscription object.
	 */
	protected function clear_caches( &$subscription ) {
		clean_post_cache( $subscription->get_id() );
		// Delete shop subscription transients.
		masteriyo( 'cache' )->delete( 'masteriyo-subscription-items-' . $subscription->get_id(), 'masteriyo-subscriptions' );
	}

	/**
	 * Get a title for the new post type.
	 *
	 * @since 2.6.10
	 *
	 * @return string
	 */
	protected function get_subscription_title() {
		// phpcs:enable
		/* translators: %s: Subscription date */
		return sprintf( __( 'Subscription &ndash; %s', 'learning-management-system' ), strftime( _x( '%1$b %2$d, %Y @ %I:%M %p', 'Subscription date parsed by strftime', 'learning-management-system' ) ) );
		// phpcs:disable
	}

	/**
	 * Get subscription key.
	 *
	 * @since 2.6.10
	 *
	 * @param \Masteriyo\Pro\Models\Subscription $subscription Subscription object.
	 * @return string
	 */
	protected function get_order_key( $subscription ) {
		return masteriyo_generate_order_key();
	}

	/**
	 * Read subscription items of a specific type from the database for this order.
	 *
	 * @since 2.6.10
	 *
	 * @param  \Masteriyo\Pro\Models\Subscription $subscription Order object.
	 * @param  string $type Order/Subscription item type.
	 * @return array
	 */
	public function read_items( $subscription, $type ) {
		$type        = trim( $type );
		$subscription_items = array();

		if ( ! empty( $subscription_items ) ) {
			return $subscription_items;
		}

		// Fetch from database.
		$subscription_items = masteriyo_get_order_items(
			array(
				'order_id' => $subscription->get_parent_id(),
				'type'     => $type,
			)
		);

		return $subscription_items;
	}
}
