<?php
/**
 * UserRegistrationAdvancedField Functions.
 *
 * General functions available on both the front-end and admin.
 *
 * @package UserRegistrationAdvancedField/Functions
 */

/**
 * Get advanced fields list.
 */
function user_registration_list_advanced_fields() {
		return apply_filters(
			'user_registration_advanced_fields',
			array(
				'section_title',
				'html',
				'timepicker',
				'phone',
				'wysiwyg',
				'select2',
				'multi_select2',
				'profile_picture',
				'range',
				'custom_url',
				'hidden',
			)
		);
}

/**
 * Valid file types.
 */
function uraf_get_valid_file_type() {

	return apply_filters(
		'user_registration_profile_picture_valid_file_type',
		array(
			'image/jpeg' => __( 'jpg/jpeg', 'user-registration-advanced-fields' ),
			'image/gif'  => __( 'gif', 'user-registration-advanced-fields' ),
			'image/png'  => __( 'png', 'user-registration-advanced-fields' ),
		)
	);
}


/**
 * Check advanced field compatibility
 */
function uraf_is_compatible() {

	$ur_pro_plugins_path = WP_PLUGIN_DIR . URAF_DS . 'user-registration-pro' . URAF_DS . 'user-registration.php';

	if ( ! file_exists( $ur_pro_plugins_path ) ) {
			return;
	}

	$ur_pro_plugin_file_path = 'user-registration-pro/user-registration.php';

	include_once ABSPATH . 'wp-admin/includes/plugin.php';

	if ( ! is_plugin_active( $ur_pro_plugin_file_path ) ) {
		return;
	}

	if ( function_exists( 'UR' ) ) {
		$user_registration_version = UR()->version;
	} else {
		$user_registration_version = get_option( 'user_registration_version' );
	}

	if ( version_compare( $user_registration_version, '4.0.0', '<' ) ) {

		return;
	}

	return 'YES';
}

/**
 * Checks Plugin Compatibility.
 */
function uraf_check_plugin_compatibility() {
	add_action( 'admin_notices', 'user_registration_advanced_fields_admin_notice', 10 );
}

/**
 * Advanced Field Admin notices.
 */
function user_registration_advanced_fields_admin_notice() {

	$class   = 'error notice is-dismissible';
	$message = uraf_is_compatible();

	if ( 'YES' !== $message ) {
		printf( '<div class="' . esc_attr( $class ) . '"><p>' . sprintf( esc_html__( 'User Registration Advance Fields requires %s version 4.0.0 or greater to work', 'user-registration-advanced-fields' ), '<a href="https://wpuserregistration.com/" target="_blank">' . esc_html__( 'User Registration Pro', 'user-registration-advanced-fields' ) . '</a>' ) . '</p></div>' );

	}
}

add_filter( 'user_registration_field_keys', 'uraf_advanced_field_keys', 10, 2 );

/**
 * Deprecate plugin missing notice.
 *
 * @deprecated 1.3.5
 *
 * @return void
 */
function uraf_admin_notices() {
	ur_deprecated_function( 'uraf_admin_notices', '1.3.5', 'user_registration_advanced_fields_admin_notice' );
}

/**
 * Hook into core to render advanced fields
 *
 * @param  string $field_type Field Type.
 * @param  string $field_key Field Key.
 * @return string
 */
function uraf_advanced_field_keys( $field_type, $field_key ) {
	switch ( $field_key ) {
		case 'section_title':
			$field_type = 'section_title';
			break;
		case 'html':
			$field_type = 'html';
			break;
		case 'wysiwyg':
			$field_type = 'wysiwyg';
			break;
		case 'timepicker':
			$field_type = 'timepicker';
			break;
		case 'phone':
			$field_type = 'phone';
			break;
		case 'select2':
			$field_type = 'select2';
			break;
		case 'multi_select2':
			$field_type = 'multi_select2';
			break;
		case 'profile_picture':
			$field_type = 'profile_picture';
			break;
		case 'range':
			$field_type = 'range';
			break;
		case 'custom_url':
			$field_type = 'custom_url';
			break;
		case 'hidden':
			$field_type = 'hidden';
			break;
	}

	return $field_type;
}

/**
 * Get wp editor.
 *
 * @param array $args Arguments.
 */
function uraf_get_wp_editor( $args ) {
	$value    = isset( $args['value'] ) ? $args['value'] : '';
	$settings = array(
		'media_buttons' => false,
		'editor_class'  => 'wysiwyg input-text ' . implode( ' ', $args['input_class'] ),
		'textarea_name' => $args['id'],
	);
	ob_start();
	wp_editor( $value, $args['id'] . '_wp_editor', $settings );
	return ob_get_clean();
}

add_filter( 'user_registration_sanitize_field', 'uraf_sanitize_fields', 10, 2 );

/**
 * Sanitize advanced fields on frontend submit
 *
 * @param  array  $form_data Form Data.
 * @param  string $field_key Field Key.
 * @return array
 */
function uraf_sanitize_fields( $form_data, $field_key ) {
	switch ( $field_key ) {
		case 'wysiwyg':
			$form_data->value = wp_kses_post( $form_data->value );
			break;
		case 'phone':
			$form_data->value = sanitize_text_field( $form_data->value );
			break;
		case 'timepicker':
			$form_data->value = sanitize_text_field( $form_data->value );
			break;
		case 'section_title':
		case 'html':
			$form_data->value = sanitize_text_field( $form_data->value );
			break;
		case 'custom_url':
			$form_data->value = sanitize_text_field( $form_data->value );
			break;
	}
		return $form_data;
}

$uraf_fields = array( 'html', 'phone', 'select2', 'multi_select2', 'section_title', 'profile_picture', 'range', 'custom_url', 'wysiwyg', 'timepicker', 'hidden' );
foreach ( $uraf_fields as $field ) {
	add_filter( 'user_registration_profile_account_filter_' . $field, 'uraf_profile_fields_data', 10, 2 );
}

/**
 * Hook field type for access in profile page
 *
 * @param  array $filter_data  Filter Data.
 * @param int   $form_id Form Id.
 */
function uraf_profile_fields_data( $filter_data, $form_id ) {
	$field       = $filter_data['field'];
	$field_name  = isset( $field->general_setting->field_name ) ? $field->general_setting->field_name : '';
	$field_index = 'user_registration_' . $field_name;
	$field_key   = isset( $field->field_key ) ? $field->field_key : '';

	if ( 'profile_picture' !== $field_key ) {
		$filter_data['fields'][ $field_index ]['description'] = ur_string_translation( $form_id, 'user_registration_' . $field_name . '_description', isset( $field->general_setting->description ) ? $field->general_setting->description : '' );
		$filter_data['fields'][ $field_index ]['type']        = $field_key;

		$filter_data['fields'][ $field_index ]['label']     = ur_string_translation( $form_id, 'user_registration_' . $field_name . '_label', isset( $field->general_setting->label ) ? $field->general_setting->label : '' );
		$filter_data['fields'][ $field_index ]['field_key'] = $field_key;
		$filter_data['fields'][ $field_index ]['required']  = isset( $field->required ) ? $field->required : '';
	}
	if ( 'html' === $field_key ) {
		$filter_data['fields'][ $field_index ]['html'] = isset( $field->general_setting->html ) ? $field->general_setting->html : '';
	} elseif ( 'select2' === $field_key || 'multi_select2' === $field_key ) {
		$filter_data['fields'][ $field_index ]['options']       = isset( $field->general_setting->options ) ? $field->general_setting->options : '';
		$filter_data['fields'][ $field_index ]['default_value'] = isset( $field->general_setting->default_value ) ? $field->general_setting->default_value : '';
	} elseif ( 'phone' === $field_key ) {
		$filter_data['fields'][ $field_index ]['input_mask'] = isset( $field->general_setting->input_mask ) ? $field->general_setting->input_mask : '(999) 999-9999';
	} elseif ( 'section_title' === $field_key ) {
		$filter_data['fields'][ $field_index ]['header_attribute'] = isset( $field->general_setting->header_attribute ) ? $field->general_setting->header_attribute : 'h3';
	}

	return $filter_data;
}

add_filter( 'user_registration_before_register_user_filter', 'strip_title_html_data_save', 10, 2 );

/**
 * Strip title and html section to save in usermeta
 *
 * @param  object $valid_form_data Form Data.
 * @param  int    $form_id Formid.
 * @return object
 */
function strip_title_html_data_save( $valid_form_data, $form_id ) {
	$strip_fields = array( 'section_title', 'html' );
	foreach ( $valid_form_data as $key => $form_data ) {
		if ( isset( $form_data->extra_params['field_key'] ) && in_array( $form_data->extra_params['field_key'], $strip_fields ) ) {
			unset( $valid_form_data[ $key ] );
		}
	}

	return $valid_form_data;
}

add_filter( 'user_registration_one_time_draggable_form_fields', 'ur_profile_picture_one_time_drag', 10, 1 );

/**
 * Make profile picture one time draggable.
 *
 * @param array $fields One time draggable fields.
 *
 * @return array One time draggable fields.
 */
function ur_profile_picture_one_time_drag( $fields ) {
	$fields[] = 'profile_picture';
	return $fields;
}

if ( ! function_exists( 'ur_time_format_options' ) ) {
	/**
	 * Time Format for advanced settings
	 *
	 * @return array
	 */
	function ur_time_format_options() {

		return apply_filters(
			'user_registration_time_format_options',
			array(
				'h:i K' => __( '12 H ', 'user-registration-advanced-fields' ),
				'H:i'   => __( '24 H', 'user-registration-advanced-fields' ),
			)
		);
	}
}
