<?php
/**
 * UserRegistrationFileUpload Settings
 *
 * @class    URFU_Settings_File
 * @version  1.0.0
 * @package  UserRegistrationFileUpload/Admin
 * @category Admin
 * @author   WPEverest
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! class_exists( 'URFU_Settings_File ' ) ) :

	/**
	 * URFU_Settings_File Class
	 */
	class URFU_Settings_File extends UR_Settings_Page {

		/**
		 * Setting Id.
		 *
		 * @var string
		 */
		public $id = 'file_upload';

		/**
		 * Constructor.
		 */
		public function __construct() {

			$this->id    = 'file_upload';
			$this->label = __( 'File Uploads', 'user-registration-file-upload' );
			add_filter( 'user_registration_settings_tabs_array', array( $this, 'add_settings_page' ), 20 );
			add_action( 'user_registration_sections_' . $this->id, array( $this, 'output_sections' ) );
			add_action( 'user_registration_settings_' . $this->id, array( $this, 'output' ) );
			add_action( 'user_registration_update_options', array( $this, 'urfu_show_notice_in_settings' ) );
			add_action( 'user_registration_settings_save_' . $this->id, array( $this, 'save' ) );
			add_action( 'user_registration_settings_file_upload', array( $this, 'urfu_file_upload_content' ) );
			add_filter( 'show_user_registration_setting_message', array( $this, 'urfu_setting_message_show' ) );
			add_filter(
				'user_registration_admin_settings_sanitize_option_user_registration_file_upload_setting_max_file_size',
				array( $this, 'sanitize_max_file_size' )
			);
		}

		/**
		 * Show setting message.
		 */
		public function urfu_show_notice_in_settings() {
			$max_upload_size_ini = wp_max_upload_size() / 1024;
			$max_file_size = get_option( 'user_registration_file_upload_setting_max_file_size');

			if ( $max_file_size > $max_upload_size_ini ) {
				UR_Admin_Settings::add_error( __("Maximum file size allowed for upload is greater than " . $max_upload_size_ini . " KB set in php.ini configuration." , 'user-registration-file-upload') );
			}
		}

		/**
		 * Show setting message.
		 *
		 * @return bool
		 */
		public function urfu_setting_message_show() {

			global $current_section;

			if ( 'uploaded_data' === $current_section ) {

				return false;
			}

			return true;
		}

		/**
		 * Content for file download setting.
		 */
		public function urfu_file_upload_content() {

			$current_section = empty( $_REQUEST['section'] ) ? '' : sanitize_title( wp_unslash( $_REQUEST['section'] ) );

			if ( 'uploaded_data' !== $current_section ) {

				return;
			}

			add_filter( 'user_registration_setting_save_label', array( $this, 'urfu_setting_label' ) );
		}

		/**
		 * Label for Save button.
		 *
		 * @return string
		 */
		public function urfu_setting_label() {

			return esc_html__( 'Download Attachments', 'user-registration-file-upload' );
		}


		/**
		 * Get sections.
		 *
		 * @return array
		 */
		public function get_sections() {
			$sections = array(
				''              => __( 'Upload Settings', 'user-registration-file-upload' ),
				'uploaded_data' => __( 'Uploaded Files', 'user-registration-file-upload' ),
			);

			return apply_filters( 'user_registration_get_sections_' . $this->id, $sections );
		}

		/**
		 * Get settings.
		 *
		 * @param string $current_section Retrieves current section.
		 * @return array
		 */
		public function get_settings( $current_section = '' ) {

			if ( 'uploaded_data' === $current_section ) {

				$settings = apply_filters(
					'user_registration_file_upload_settings',
					array(
						'title'    => __( 'Uploaded Files', 'user-registration-file-upload' ),
						'sections' => array(
							'user_registration_uploaded_data_download' => array(
								'title'    => __( 'Download Uploaded Files', 'user-registration-file-upload' ),
								'type'     => 'card',
								'desc'     => '',
								'settings' => array(
									array(
										'title'    => __( 'Select Registration Forms', 'user-registration-file-upload' ),
										'desc'     => __( 'Choose registration forms for attachment downloads.', 'user-registration-file-upload' ),
										'id'       => 'user_registration_file_upload_form_id_for_attachment_download',
										'default'  => array_keys( urfu_get_all_user_registration_form() ),
										'type'     => 'multiselect',
										'class'    => 'ur-enhanced-select',
										'css'      => 'min-width: 350px;',
										'desc_tip' => true,
										'options'  => urfu_get_all_user_registration_form(),
									),
								),
							),
						),
					)
				);

			} else {
				$settings = urfu_file_upload_settings();
			}

			return apply_filters( 'user_registration_file_upload_settings' . $this->id, $settings );
		}

		/**
		 * Outputs File upload settings Page
		 *
		 * @return void
		 */
		public function output() {

			global $current_section;

			$settings = $this->get_settings( $current_section );

			UR_Admin_Settings::output_fields( $settings );

		}

		/**
		 * Save settings
		 */
		public function save() {

			global $current_section;

			$settings = $this->get_settings( $current_section );

			UR_Admin_Settings::save_fields( $settings );


		}

		/**
		 * Prevent setting invalid filesize value.
		 *
		 * @since 2.2.6
		 * @param [int] $value Max Allowed File Size.
		 * @return int
		 */
		public function sanitize_max_file_size( $value ) {
			return absint( $value );
		}
	}

endif;

return new URFU_Settings_File();
