<?php
/**
 * UserRegistrationFileUpload Frontend.
 *
 * @class    URFU_Frontend
 * @version  1.0.0
 * @package  UserRegistrationFileUpload/Admin
 * @category Admin
 * @author   WPEverest
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * URFU_Frontend Class
 */
class URFU_Frontend {

	/**
	 * Hook in tabs.
	 */
	public function __construct() {

		if ( 'YES' !== urfu_is_compatible() ) {
			return;
		}
		add_filter( 'user_registration_form_field_file_path', 'urfu_form_field_file', 10, 1 );
		add_filter( 'user_registration_registered_form_fields', 'urfu_registered_form_fields', 10, 1 );
		add_filter( 'user_registration_form_params', array( $this, 'user_registration_form_params' ), 10, 1 );
		add_action( 'user_registration_enqueue_scripts', array( $this, 'user_registration_file_upload_scripts' ), 10, 2 );
		add_action( 'user_registration_my_account_enqueue_scripts', array( $this, 'user_registration_file_upload_scripts' ), 10, 2 );
		add_filter( 'user_registration_form_field_file', array( $this, 'user_registration_form_field_file' ), 10, 4 );
		add_action( 'user_registration_after_user_meta_update', array( $this, 'user_registration_after_user_meta_update' ), 10, 3 );
		add_filter( 'user_registration_before_save_profile_details', array( $this, 'user_registration_before_save_profile_details' ), 10, 3 );
		add_filter( 'user_registration_profile_update_response', array( $this, 'user_registration_attachments_after_ajax_submission_in_profile_details' ), 10, 1 );
	}

	/**
	 * It returns the attachments id after ajax submission in profile details.
	 *
	 * @param [int] $user_id User ID.
	 * @param [int] $form_id Form ID.
	 */
	public function user_registration_attachments_after_ajax_submission_in_profile_details( $response ) {

		if ( ! ur_string_to_bool( get_option( 'user_registration_ajax_form_submission_on_edit_profile', false ) ) ) {
			return $response;
		}

		$user_id = get_current_user_id();
		$form_id = get_user_meta( $user_id, 'ur_form_id', true );

		$post_content = ur_get_post_content( $form_id );
		$files        = ur_get_form_data_by_key( $post_content, 'file' );

		$file_upload_attachments = array();

		foreach ( $files as $field_name => $value ) {
			$attachment_ids = get_user_meta( $user_id, 'user_registration_' . $field_name );
			$attachment_ids = is_array( $attachment_ids ) ? explode( ',', $attachment_ids[0] ) : explode( ',', $attachment_ids );
			$tmp_array      = array();

			foreach ( $attachment_ids as $key => $attachment_id ) {
				$filename = basename( get_attached_file( $attachment_id ) ); // Just the file name

				$tmp_array[ $attachment_id ] = array(
					'url'      => wp_get_attachment_url( $attachment_id ),
					'filename' => $filename,
				);
			}

			$file_upload_attachments[ $field_name ] = $tmp_array;
		}

		$response['file_upload_attachments'] = $file_upload_attachments;
		return $response;
	}

	/**
	 * Upload Files while edit profile saved.
	 *
	 * @param array $profile Profile Data.
	 * @param int   $user_id User ID.
	 * @param int   $form_id Form ID.
	 */
	public function user_registration_before_save_profile_details( $profile, $user_id, $form_id ) {
		$post_content = ur_get_post_content( $form_id );
		$files        = ur_get_form_data_by_key( $post_content, 'file' );
		$upload_dir   = wp_upload_dir();
		$upload_path  = UR_UPLOAD_PATH . 'file-uploads'; /*Get path of upload dir of User Registration File Uploads*/

		// Checks if the upload directory exists and create one if not.
		if ( ! file_exists( $upload_path ) ) {
			wp_mkdir_p( $upload_path );
		}

		$valid_form_data = array();

		foreach ( $files as $field_name => $value ) {
			$previous_files = get_user_meta( $user_id, 'user_registration_' . $field_name );
			$previous_files = isset( $previous_files[0] ) ? explode( ',', $previous_files[0] ) : array();
			if ( ! ur_string_to_bool( get_option( 'user_registration_ajax_form_submission_on_edit_profile', false ) ) ) {
				$removed_files = isset( $_POST[ 'urfu_removed_files_' . $field_name ] ) ? (array) json_decode( stripslashes( $_POST[ 'urfu_removed_files_' . $field_name ] ), true ) : array();
				if ( ! empty( $previous_files ) && ! empty( $removed_files ) ) {
					foreach ( $previous_files as $key => $attachment_id ) {

						if ( in_array( $attachment_id, $removed_files ) ) {
							$file = get_attached_file( $attachment_id );

							if ( $file ) {
								unset( $previous_files[ $key ] );
								unlink( $file );
								wp_delete_attachment( $attachment_id, true );
							}
						}
					}
				}
				$file_upload  = isset( $_POST[ 'urfu_uploaded_file_' . $field_name ] ) && ! empty( $_POST[ 'urfu_uploaded_file_' . $field_name ] ) && is_array( $_POST[ 'urfu_uploaded_file_' . $field_name ] ) ? implode( ',', $_POST[ 'urfu_uploaded_file_' . $field_name ] ) : '';
				$file_upload .= ! empty( $previous_files ) ? ! empty( $file_upload ) ? ',' . implode( ',', $previous_files ) : '' . implode( ',', $previous_files ) : '';

				$valid_form_data[ $field_name ]        = new stdClass();
				$valid_form_data[ $field_name ]->value = $file_upload;
			} elseif ( isset( $_POST['form_data'] ) ) {
					$form_data = json_decode( wp_unslash( $_POST['form_data'] ) ); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized

				foreach ( $form_data as $data ) {
					$data->field_name = ( strpos( $data->field_name, 'user_registration_' ) === 0 ) ? substr( $data->field_name, 18 ) : $data->field_name;

					if ( $field_name == $data->field_name ) {
						$removed_files = isset( $_POST[ 'urfu_removed_files_' . $field_name ] ) ? (array) json_decode( stripslashes( $_POST[ 'urfu_removed_files_' . $field_name ] ), true ) : array();
						if ( ! empty( $previous_files ) && ! empty( $removed_files ) ) {
							foreach ( $previous_files as $key => $attachment_id ) {

								if ( in_array( $attachment_id, $removed_files ) ) {
									$file = get_attached_file( $attachment_id );

									if ( $file ) {
										unset( $previous_files[ $key ] );
										unlink( $file );
										wp_delete_attachment( $attachment_id, true );
									}
								}
							}
						}

						$file_upload  = isset( $data->value ) && ! empty( $data->value ) && is_array( $data->value ) ? implode( ',', $data->value ) : '';
						$file_upload .= ! empty( $previous_files ) ? ! empty( $file_upload ) ? ',' . implode( ',', $previous_files ) : '' . implode( ',', $previous_files ) : '';

						$valid_form_data[ $field_name ]        = new stdClass();
						$valid_form_data[ $field_name ]->value = $file_upload;
					}
				}
			}
			unset( $profile[ 'user_registration_' . $field_name ] );
		}
		$this->urfu_upload_files( $files, $valid_form_data, $upload_path, $user_id, $form_id );
		return $profile;
	}

	/**
	 * Upload file on form submit.
	 *
	 * @param [array] $valid_form_data Form Data
	 * @param [int]   $form_id Form ID.
	 * @param [int]   $user_id User ID.
	 */
	public function user_registration_after_user_meta_update( $valid_form_data, $form_id, $user_id ) {
		$post_content = ur_get_post_content( $form_id );
		$files        = ur_get_form_data_by_key( $post_content, 'file' );
		$upload_dir   = wp_upload_dir();

		$upload_path = apply_filters( 'user_registration_file_upload_url', UR_UPLOAD_PATH . 'file-uploads' ); /*Get path of upload dir of WordPress*/

		// Checks if the upload directory exists and create one if not.
		if ( ! file_exists( $upload_path ) ) {
			wp_mkdir_p( $upload_path );
		}
		$this->urfu_upload_files( $files, $valid_form_data, $upload_path, $user_id, $form_id );
	}

	public function urfu_upload_files( $files, $valid_form_data, $upload_path, $user_id, $form_id ) {

		$upload_path = trailingslashit( $upload_path );
		if ( ! empty( $files ) ) {
			foreach ( $files as $field_name => $value ) {

				$data             = isset( $valid_form_data[ $field_name ]->value ) && ! empty( $valid_form_data[ $field_name ]->value ) ? ( ! is_array( $valid_form_data[ $field_name ]->value ) ? explode( ',', $valid_form_data[ $field_name ]->value ) : $valid_form_data[ $field_name ]->value ) : array();
				$attachment_id    = array();
				$attachments_path = array();

				foreach ( $data as $key => $upload_file ) {
					if ( ! is_numeric( $upload_file ) ) {
						$upload = maybe_unserialize( crypt_the_string( $upload_file, 'd' ) );

						if ( isset( $upload['file_name'] ) && isset( $upload['file_path'] ) && isset( $upload['file_extension'] ) ) {
							$file_name = wp_unique_filename( $upload_path, $upload['file_name'] );
							$file_path = $upload_path . sanitize_file_name( $file_name );
							// Check the type of file. We'll use this as the 'post_mime_type'.
							$filetype = wp_check_filetype( basename( $file_name ), null );
							$moved    = rename( $upload['file_path'], $file_path );

							if ( $moved ) {
								$attachment_id[ $key ] = wp_insert_attachment(
									array(
										'guid'           => $file_path,
										'post_mime_type' => $filetype['type'],
										'post_title'     => preg_replace( '/\.[^.]+$/', '', sanitize_file_name( $file_name ) ),
										'post_content'   => '',
										'post_status'    => 'inherit',
									),
									$file_path
								);

								if ( ! is_wp_error( $attachment_id[ $key ] ) ) {
									include_once ABSPATH . 'wp-admin/includes/image.php';

									$attachments_path[] = $file_path;

									// Generate and save the attachment metas into the database.
									wp_update_attachment_metadata( $attachment_id[ $key ], wp_generate_attachment_metadata( $attachment_id[ $key ], $file_path ) );
								}
							}
						}
					} else {
						$attachment_id[ $key ] = $upload_file;
					}
				}

				$attachment_id = implode( ',', $attachment_id );

				update_user_meta( $user_id, 'user_registration_' . $field_name, $attachment_id );

				if ( ! empty( $attachment_id ) ) {
					do_action( 'user_registration_after_file_upload', $upload_path, $attachment_id, $user_id, $form_id, $attachments_path );
				}
			}
		}
	}

	/**
	 * Renders the file upload field in the frontend.
	 *
	 * @param string $field List of fields.
	 * @param string $key Field key.
	 * @param array  $args Fields settings values.
	 * @param string $value Value saved in a field for a user.
	 */
	public function user_registration_form_field_file( $field, $key, $args, $value ) {

		$value             = ! empty( $value ) ? $value : '';
		$value             = isset( $args['value'] ) ? $args['value'] : $value;
		$custom_attributes = array();

		if ( ! empty( $args['custom_attributes'] ) && is_array( $args['custom_attributes'] ) ) {
			foreach ( $args['custom_attributes'] as $attribute => $attribute_value ) {
				$custom_attributes[] = esc_attr( $attribute ) . '="' . esc_attr( $attribute_value ) . '"';
			}
		}

		/* Conditional Logic codes */
		$rules                      = array();
		$rules['conditional_rules'] = isset( $args['conditional_rules'] ) ? $args['conditional_rules'] : '';
		$rules['logic_gate']        = isset( $args['logic_gate'] ) ? $args['logic_gate'] : '';
		$rules['rules']             = isset( $args['rules'] ) ? $args['rules'] : array();
		$rules['required']          = isset( $args['required'] ) ? $args['required'] : '';

		foreach ( $rules['rules'] as $rules_key => $rule ) {
			if ( empty( $rule['field'] ) ) {
				unset( $rules['rules'][ $rules_key ] );
			}
		}

		$rules['rules'] = array_values( $rules['rules'] );

		$rules = ( ! empty( $rules['rules'] ) && isset( $args['enable_conditional_logic'] ) ) ? wp_json_encode( $rules ) : '';
		/*Conditonal Logic codes end*/

		$is_required = isset( $args['required'] ) ? $args['required'] : 0;

		ob_start();
		?>
		<div class="urfu-file-upload" data-id="<?php echo esc_attr( $args['id'] ); ?>">
			<div class="form-row " id="<?php echo esc_attr( $key ); ?>_field" data-priority="">
				<label for="<?php echo esc_attr( $key ); ?>" class="ur-label"><?php echo esc_html( $args['label'] ); ?>
				<?php
				if ( $is_required ) {
					?>
					<abbr class="required"
						title="required">*
					</abbr>
					<?php
				}

				if ( isset( $args['tooltip'] ) && ur_string_to_bool( $args['tooltip'] ) ) {
					echo ur_help_tip( $args['tooltip_message'], false, 'ur-portal-tooltip' );
				}
				?>
				</label>
				<?php
				$attachment_ids = explode( ',', $value );
				$this->dropzone_file_upload_container( $rules, $args, $key, $value, $attachment_ids, $custom_attributes );

				if ( $args['description'] ) {
					echo '<span class="description">' . $args['description'] . '</span>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
				}
				?>
			</div>
		</div>

		<?php
		$field_content = ob_get_clean();

		if ( $args['return'] ) {
			return $field_content;
		} else {
			echo $field_content; //PHPCS:ignore;
			return '';
		}
	}

	/**
	 * Initializes the dropzoe container and handles the file upload.
	 *
	 * @param array  $rules Consitioal logic rules for the field.
	 * @param array  $args Fields settings values.
	 * @param string $key Field key.
	 * @param string $value Value saved in a field for a user.
	 * @param array  $attachment_ids Array of attachment id of respective files.
	 * @param array  $custom_attributes Extra attribues.
	 */
	private function dropzone_file_upload_container( $rules, $args, $key, $value, $attachment_ids, $custom_attributes ) {
		$hide_class = '';

		if ( isset( $args['max_files'] ) && '' !== $args['max_files'] ) {
			if ( ! empty( $attachment_ids ) && ! empty( $value ) ) {
				$max_files = $args['max_files'] - count( $attachment_ids );
			} else {
				$max_files = $args['max_files'];
			}

			if ( $max_files <= 0 ) {
				$hide_class = 'hide';
			}
		} else {
			$max_files = 1;
		}

		$max_upload_size_ini           = wp_max_upload_size();
		$max_upload_size_ini_kb        = (int) $max_upload_size_ini / 1024;
		$max_upload_size_options_value = get_option( 'user_registration_file_upload_setting_max_file_size', $max_upload_size_ini_kb );

		if ( isset( $args['max_upload_size'] ) && '' !== $args['max_upload_size'] ) {
			$max_upload_size_options_value = $args['max_upload_size'];
		}

		// Validates if the uploaded file has acceptable upload size.
		if ( $max_upload_size_options_value > $max_upload_size_ini_kb ) {
			$max_upload_size_options_value = $max_upload_size_ini_kb;
		}

		?>
		<div class="dropzone dz-clickable" id="ur_<?php echo esc_attr( $args['id'] ); ?>" data-id="<?php echo esc_attr( $args['id'] ); ?>">
			<div class="dz-message <?php echo esc_attr( $hide_class ); ?>">
				<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 32 32" width="32px" height="32px" fill="#868e96"><path class="cls-1" d="M18.12,17.52,17,16.4V25a1,1,0,0,1-2,0V16.4l-1.12,1.12a1,1,0,0,1-1.42,0,1,1,0,0,1,0-1.41l2.83-2.83a1,1,0,0,1,1.42,0l2.83,2.83a1,1,0,0,1-.71,1.7A1,1,0,0,1,18.12,17.52ZM22,22H20a1,1,0,0,1,0-2h2a4,4,0,0,0,.27-8,1,1,0,0,1-.84-.57,6,6,0,0,0-11.36,1.69,1,1,0,0,1-1,.86H9A3,3,0,0,0,9,20h3a1,1,0,0,1,0,2H9a5,5,0,0,1-.75-9.94A8,8,0,0,1,23,10.1,6,6,0,0,1,22,22Z"></path></svg>
				<span class="user-registration-file-upload-title">
					<?php esc_html_e( 'Drop your file here or click here to upload', 'user-registration-file-upload' ); ?>
				</span>
				<span class="user-registration-file-upload-hint">
					<?php printf( esc_html__( 'You can upload up to %d files', 'user-registration-file-upload' ), $max_files ); ?>
				</span>
			</div>
			<span class="urfu-upload-node" style="height: 0;width: 0;margin: 0;padding: 0;float: left;border: 0;overflow: hidden;">
				<input type="<?php echo esc_attr( $args['type'] ); ?>" multiple>
				<?php
				// valid extension for file upload.
				$default_file_types = array_flip( urfu_get_valid_file_type() );

				$valid_file_type = get_option( 'user_registration_file_upload_setting_valid_file_type', array() );

				if ( empty( $valid_file_type ) ) {
					$valid_file_type = $default_file_types;
				}

				if ( ! empty( $args['valid_file_type'] ) ) {
					$valid_file_type = $args['valid_file_type'];
				}

				$valid_file_type = implode( ', ', $valid_file_type );

				echo '<input data-rules="' . esc_attr( $rules ) . '" type="text" class="urfu-file-input dropzone-input input-text ' . esc_attr( implode( ' ', $args['input_class'] ) ) . '" name="' . esc_attr( $key ) . '" id="' . esc_attr( $args['id'] ) . '" placeholder="' . esc_attr( $args['placeholder'] ) . '"  value="' . esc_attr( $value ) . '" ' . esc_attr( implode( ' ', $custom_attributes ) ) . ' data-max-file-number="' . esc_attr( $max_files ) . '" data-max-file-limit="' . esc_attr( $args['max_files'] ) . '" data-max-upload-size-limit="' . esc_attr( $max_upload_size_options_value ) . '" data-valid-file-type="' . esc_attr( $valid_file_type ) . '" />';
				?>
			</span>
			<?php
			if ( ! empty( $value ) ) {
				foreach ( $attachment_ids as $attachment_id ) {
					?>
					<div class="dz-preview dz-processing dz-image-preview dz-success dz-complete">
						<div class="dz-image">
							<?php
							$attachment_url = wp_get_attachment_url( $attachment_id );

							if ( empty( $attachment_url ) ) {
								$attachment_url = home_url() . '/wp-includes/images/media/text.png';
							}

							$filename_only = basename( get_attached_file( $attachment_id ) ); // Just the file name.
							$filesize_only = size_format( filesize( get_attached_file( $attachment_id ) ), 1 ); // Just the file size.
							$ext           = pathinfo( $attachment_url, PATHINFO_EXTENSION );
							$fileIcon      = user_registration_file_upload_check_file_types( $ext );
							$fileIcon      = ! is_null( $fileIcon ) ? URFU_PLUGIN_ASSETS . '/images/filetypes/' . $fileIcon . '.svg' : $attachment_url;
							?>

							<img data-dz-thumbnail="" alt="<?php echo esc_html( sanitize_text_field( $filename_only ) ); ?>"  src="<?php echo esc_url( $fileIcon ); ?>"/>
						</div>
						<div class="dz-details">
							<div class="dz-size">
								<span data-dz-size=""><strong><?php echo esc_html( sanitize_text_field( $filesize_only ) ); ?></strong></span>
							</div>
							<div class="dz-filename">
								<span data-dz-name=""><?php echo esc_html( sanitize_text_field( $filename_only ) ); ?></span>
							</div>
						</div>
						<a class="urfu-download-file" href="<?php echo esc_url( $attachment_url ); ?>" title="Download" target="_blank" download ><span class="dashicons dashicons-arrow-down-alt"></span></a>
						<?php
						$data_field_name = ( strpos( $args['id'], 'user_registration_' ) === 0 ) ? substr( $args['id'], 18 ) : $args['id'];

						?>
						<a class="dz-remove urfu-remove-file" href="javascript:undefined;" title="Remove" data-dz-remove="" data-attachment-id="<?php echo esc_html( sanitize_text_field( $attachment_id ) ); ?>" data-field-name="<?php echo esc_attr( $data_field_name ); ?>"></a>
					</div>
					<?php
				}
			}
			?>
		</div>
		<?php
	}

	/**
	 * Add extra params to form tag
	 *
	 * @param string $extra_params Extra params string to add.
	 */
	public function user_registration_form_params( $extra_params ) {

		return $extra_params .= ' "enctype=multipart/form-data"';
	}

	/**
	 * Enqueue File Upload Scripts.
	 *
	 * @param array $form_data_array Form data.
	 * @param int   $form_id Form Identifier.
	 */
	public function user_registration_file_upload_scripts( $form_data_array, $form_id ) {
		wp_enqueue_media();
		$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
		wp_register_script( 'user-registration-file-upload-script', URFU()->plugin_url() . '/assets/js/frontend/user-registration-file-upload-script' . $suffix . '.js', array( 'jquery' ), URFU_VERSION );
		wp_register_script( 'user-registration-dropzone-file-upload-script', URFU()->plugin_url() . '/assets/js/dropzone/dropzone' . $suffix . '.js', URFU_VERSION );
		wp_register_style( 'user-registration-file-upload-style', URFU()->plugin_url() . '/assets/css/user-registration-file-upload-style.css', array(), URFU_VERSION );
		wp_enqueue_script( 'user-registration-file-upload-script' );
		wp_enqueue_script( 'user-registration-dropzone-file-upload-script' );
		wp_enqueue_style( 'user-registration-file-upload-style' );
		wp_localize_script(
			'user-registration-file-upload-script',
			'urfu_script_data',
			array(
				'ajax_url'                => admin_url( 'admin-ajax.php' ),
				'urfu_upload_nonce'       => wp_create_nonce( 'urfu_file_upload_nonce' ),
				'urfu_uploading'          => __( 'Uploading...', 'user-registration-file-upload' ),
				'urfu_something_wrong'    => __( 'Something wrong, please try again.', 'user-registration-file-upload' ),
				'urfu_max_file_remaining' => __( 'You can upload up to %qty% files', 'user-registration-file-upload' ),
				'urfu_file_size_exceeded' => __( 'File size exceed, please upload file less than %qty% MB.', 'user-registration-file-upload' ),
				'home_url'                => URFU_PLUGIN_ASSETS,
			)
		);
	}
}

return new URFU_Frontend();
