<?php
use Automattic\WooCommerce\Blocks\Package;
use Automattic\WooCommerce\Blocks\StoreApi\Schemas\CartSchema;
use Automattic\WooCommerce\Blocks\StoreApi\Schemas\CheckoutSchema;

/**
 * Shipping Workshop Extend Store API.
 */
class URWC_Extend_Store_Endpoint {
	/**
	 * Stores Rest Extending instance.
	 *
	 * @var ExtendRestApi
	 */
	private static $extend;

	/**
	 * Plugin Identifier, unique to each plugin.
	 *
	 * @var string
	 */
	const IDENTIFIER = 'urwc-checkout-sync-field';

	/**
	 * Bootstraps the class and hooks required data.
	 */
	public static function init() {

		if ( ! function_exists( 'ur_string_to_bool' ) ) {
			return;
		}

		if ( ! ur_string_to_bool( get_option( 'user_registration_woocommrece_settings_sync_checkout', false ) ) ) {
			return;
		}
		self::$extend = Automattic\WooCommerce\StoreApi\StoreApi::container()->get( Automattic\WooCommerce\StoreApi\Schemas\ExtendSchema::class );
		self::extend_store();
	}

	/**
	 * Registers the actual data into each endpoint.
	 */
	public static function extend_store() {
		if ( is_callable( array( self::$extend, 'register_endpoint_data' ) ) ) {
			self::$extend->register_endpoint_data(
				array(
					'endpoint'        => CheckoutSchema::IDENTIFIER,
					'namespace'       => self::IDENTIFIER,
					'schema_callback' => array( 'URWC_Extend_Store_Endpoint', 'extend_checkout_schema' ),
					'schema_type'     => ARRAY_A,
				)
			);
		}
	}

	/**
	 * Register user registration checkout sync fields for server side validation schema into the Checkout endpoint.
	 *
	 * @return array Registered schema.
	 */
	public static function extend_checkout_schema() {
		$form_id               = get_option( 'user_registration_woocommerce_settings_form', 0 );
		$sync_checkout_enabled = ur_string_to_bool( get_option( 'user_registration_woocommrece_settings_sync_checkout', false ) );

		if ( 0 < $form_id && $sync_checkout_enabled ) {
			return array(
				'form-data' => array(
					'description' => 'These fields are sync in checkout block page',
					'type'        => 'object',
					'arg_options' => array(
						'validate_callback' => function ( $value ) {
							$form_data = array();

							foreach ( $value as $key => $form_value ) {
								$key = str_replace( '[]', '', $key );
								$form_data[ $key ] = $form_value;
							}
							return self::handle_backend_validation_for_ur_fields_in_checkout_block( $form_data );
						},
					),
				),
			);
		}
	}

	/**
	 * Handle backend validation for ur fields in checkout block.
	 *
	 * @param array $value Value.
	 */
	public static function handle_backend_validation_for_ur_fields_in_checkout_block( $value ) {
		$checkout_form_id            = get_option( 'user_registration_woocommerce_settings_form', 0 );
		$checkout_form_fields        = get_option( 'user_registration_woocommerce_checkout_fields', array() );
		$urcl_hide_fields            = isset( $value['urcl_hide_fields'] ) ? (array) json_decode( stripslashes( $value['urcl_hide_fields'] ), true ) : array();
		$mapped_checkout_form_fields = map_checkout_form_fields( $checkout_form_fields, $checkout_form_id );

		if ( 0 < $checkout_form_id && ! is_user_logged_in() ) {
			$profile = urwc_get_form_fields( $checkout_form_id );

			if ( ! is_wp_error( $profile ) ) {
				foreach ( $profile as $profile_key => $profile_field ) {
					$field_name = str_replace( 'user_registration_', '', $profile_key );

					if ( ! in_array( $field_name, $urcl_hide_fields, true ) ) {
						if ( in_array( $profile_key, $mapped_checkout_form_fields, true ) ) {
							$passed = self::validate_block_single_fields( $value, $profile_field, $field_name, $profile_key, $checkout_form_id );
							if ( is_wp_error( $passed ) ) {
								return $passed;
							}
						}
					}
				}
			}
		}
		if ( ! is_wp_error( $value ) ) {
			return $value;
		}
	}

	/**
	 * Validate single field block.
	 *
	 * @param array  $value Value.
	 * @param array  $profile_field Profile Field.
	 * @param string $field_name Field Name.
	 * @param string $profile_key Profile key.
	 * @param int    $form_id Form ID.
	 * @param bool   $passed Passed.
	 */
	public static function validate_block_single_fields( $value, $profile_field, $field_name, $profile_key, $form_id, $passed = true ) {

		$field_key   = $profile_field['field_key'];
		$class_name  = ur_load_form_field_class( $field_key );
		$filter_hook = "user_registration_validate_{$field_key}_message";
		$class       = class_exists( $class_name ) ? $class_name::get_instance() : '';

		if ( empty( $class ) ) {
			return new WP_Error( 'ur_invalid_data', 'Class not found. - ' . $class_name );
		}

		$single_form_field                  = new StdClass();
		$single_form_field->field_key       = $profile_field['field_key'];
		$single_form_field->field_name      = $field_name;
		$single_form_field->general_setting = $profile_field['general_setting'];
		$single_form_field->advance_setting = $profile_field['advance_setting'];

		$single_form_field->general_setting->required = '1' == $profile_field['required'] ? 'yes' : 'no';

		$form_data = (object) array(
			'value'        => isset( $value[ $profile_key ] ) ? $value[ $profile_key ] : '',
			'field_type'   => $profile_field['type'],
			'field_name'   => $field_name,
			'label'        => $profile_field['label'],
			'extra_params' => array(
				'field_key' => $profile_field['field_key'],
				'label'     => $profile_field['label'],
			),
		);

		if ( class_exists( 'UR_Form_Validation' ) ) {
			$validation_class  = new UR_Form_Validation();
			$field_validations = $validation_class->get_field_validations( $single_form_field->field_key );

			if ( $validation_class->is_field_required( $single_form_field, $form_data ) ) {
				array_unshift( $field_validations, 'required' );
			}

			if ( ! empty( $field_validations ) ) {
				if ( in_array( 'required', $field_validations, true ) || ! empty( $form_data->value ) ) {
					foreach ( $field_validations as $validation ) {
						$result = $validation_class::$validation( $form_data->value );
						if ( ! is_bool( $result ) ) {
							return $result;
						}
					}
				}
			}
		}
		$class->validation( $single_form_field, $form_data, $filter_hook, $form_id );
		$response = apply_filters( $filter_hook, '' );
		if ( ! empty( $response ) ) {
			if ( is_array( $response ) ) {
				foreach ( $value as $key => $form_value ) {
					$field_name_key = str_replace( 'user_registration_', '', $key );
					$message        = isset( $response[ $field_name_key ] ) ? $response[ $field_name_key ] : false;
					if ( $message ) {
						return new WP_Error( 'ur_invalid_data', $profile_field['label'] . ' - ' . $message );
					}
				}
			} else {
				return new WP_Error( 'ur_invalid_data', $profile_field['label'] . ' - ' . $response );
			}
		}
		return $passed;
	}
}
