<?php
/**
 * Masteriyo Mailchimp Integration addon setup.
 *
 * @package Masteriyo\Addons\MailchimpIntegration
 *
 * @since 2.7.1
 */
namespace Masteriyo\Addons\MailchimpIntegration;

use Masteriyo\Constants;
use DrewM\MailChimp\MailChimp;
use Masteriyo\Addons\MailchimpIntegration\Setting;
use Masteriyo\Addons\MailchimpIntegration\RestApi\Controllers\Version1\MailchimpIntegrationController;
use Masteriyo\Pro\Addons;

defined( 'ABSPATH' ) || exit;

/**
* Main Masteriyo Mailchimp Integration class.
*
*/
class MailchimpIntegrationAddon {

	/**
	 * Instance
	 *
	 * @since 2.7.1
	 *
	 * @var \Masteriyo\Addons\MailchimpIntegration\MailchimpIntegrationAddon
	 */
	protected static $instance = null;

	/**
	 * Instance of Setting class.
	 *
	 * @since 2.7.1
	 * @var Masteriyo\Addons\MailchimpIntegration\Setting
	 */
	public $setting = null;

	/**
	 * Initialize module.
	 *
	 * @since 2.7.1
	 */
	public function init() {
		$this->setting = new Setting();
		$this->setting->init();
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 2.7.1
	 */
	public function init_hooks() {
		add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );
		add_action( 'masteriyo_mailchimp_consent_checkbox', array( $this, 'render_consent_message_for_mailchimp' ) );
		add_action( 'masteriyo_created_customer', array( $this, 'mailchimp_email_insertion_on_registration' ), 3, 3 );
	}


	/**
	 * Consent Message for the user.
	 *
	 * @since 2.7.1
	 *
	 * @return template for mailchimp integration
	 */
	public function render_consent_message_for_mailchimp() {
		$setting         = $this->setting;
		$data            = $setting->get();
		$consent_message = $data['subscriber_consent_message'];

		if ( false === $data['forced_email_subscription'] || ! ( new Addons() )->is_active( MASTERIYO_MAILCHIMP_INTEGRATION_SLUG ) ) {
			return include_once Constants::get( 'MASTERIYO_MAILCHIMP_INTEGRATION_TEMPLATES' ) . '/consent-form-mailchimp.php';
		}

	}

	/**
	 * Register rest routes.
	 *
	 * @since 2.7.1
	 */
	public function register_rest_routes() {
		$controller = masteriyo( MailchimpIntegrationController::class );

		if ( $controller ) {
			$controller->register_routes();
		}
	}

	/**
	 * Email insertion when user is registered
	 *
	 * @since 2.7.1
	 *
	 * @param Masteriyo\Models\User $user
	 */
	public function mailchimp_email_insertion_on_registration( $user, $password_generated, $args ) {
		$setting = $this->setting;
		$data    = $setting->get();
		$consent = masteriyo_array_get( $args, 'mailchimp-permission-check', false );
		if ( ! $setting->get( 'forced_email_subscription' ) && ! $consent ) {
			return;
		}

		$mailchimp = new MailChimp( $setting->get( 'api_key' ) );
		// lists of all the list id and names
		$list_of_audience = $mailchimp->get( 'lists?fields=lists.id,lists.name' );

		$user_email = $user->get_email();
		$user_name  = $user->get_username();
		$first_name = $user->get_first_name();
		$last_name  = $user->get_last_name();

		$list_id = $list_of_audience['lists'][0]['id'];

		$mailchimp_interested_list = array();
		$interests                 = $setting->get( 'default_group.interests' );
		foreach ( $interests as $interest ) {
			$mailchimp_interested_list[ $interest['value'] ] = true;
		}

		// Insert emails to the mailchimp accounts.
		$mailchimp->post(
			"lists/{$list_id}/members",
			array(
				'email_address' => $user_email,
				'status'        => 'subscribed',
				'merge_fields'  => array(
					'FNAME'    => $first_name,
					'LNAME'    => $last_name,
					'USERNAME' => $user_name,
				),
				'interests'     => $mailchimp_interested_list,
			)
		);

		if ( $mailchimp->success() ) {
			return array(
				'success' => true,
			);
		} else {
			return array(
				'error message' => $mailchimp->getLastError(),
			);
		}
	}


	/**
	 * Get all lists and groups with is the single object.
	 *
	 * @since 2.7.1
	 *
	 * @return array
	 */
	public function get_mailchimp_audience_and_groups() {
		$setting = $this->setting;
		$error   = new \WP_Error();
		try {
			$mailchimp = new MailChimp( $setting->get( 'api_key' ) );
		} catch ( \Exception $e ) {
			return new \WP_Error(
				__( 'Sorry, something went wrong.', 'learning-management-system' ),
				$e->getMessage(),
				array(
					'status' => 403,
				)
			);
		}

		$result = $mailchimp->get( 'lists' );

		$audiences = array();
		if ( isset( $result['lists'] ) && ! empty( $result['lists'] ) ) {
			foreach ( $result['lists'] as $list ) {
				$audience = array(
					'value'  => $list['id'],
					'label'  => $list['name'],
					'groups' => array(),
				);

				// Get all the groups in the list
				$groups_result = $mailchimp->get( 'lists/' . $list['id'] . '/interest-categories' );
				if ( isset( $groups_result['categories'] ) && ! empty( $groups_result['categories'] ) ) {
					foreach ( $groups_result['categories'] as $group ) {
						$interests        = array();
						$interests_result = $mailchimp->get( 'lists/' . $list['id'] . '/interest-categories/' . $group['id'] . '/interests' );
						foreach ( $interests_result['interests'] as $interest ) {
							$interests[] = array(
								'value' => $interest['id'],
								'label' => $interest['name'],
							);
						}
						$audience['groups'][] = array(
							'value'     => $group['id'],
							'label'     => $group['title'],
							'interests' => $interests,
						);
					}
				} else {
					return array(
						'groups_result' => isset( $groups_result['detail'] ) ? $groups_result['detail'] : '',
					);
				}

				array_push( $audiences, $audience );
			}
		} else {
			return new \WP_Error(
				__( 'Sorry, something went wrong.', 'learning-management-system' ),
				isset( $result['detail'] ) ? $result['detail'] : '',
				array(
					'status' => 403,
				)
			);
		}
		return $audiences;

	}
}


