<?php
/**
 * Mailchimp Integration rest controller.
 *
 * @since 2.7.1
 *
 * @package Masteriyo\Addons
 * @subpackage Masteriyo\Addons\MailchimpIntegration
 */

namespace Masteriyo\Addons\MailchimpIntegration\RestApi\Controllers\Version1;

defined( 'ABSPATH' ) || exit;

use Masteriyo\Addons\MailchimpIntegration\MailchimpIntegrationAddon;

use Masteriyo\Helper\Permission;
use Masteriyo\RestApi\Controllers\Version1\CrudController;
use Masteriyo\Addons\MailchimpIntegration\Setting;

class MailchimpIntegrationController extends CrudController {
	/**
	 * Endpoint namespace.
	 *
	 * @var string
	 */
	protected $namespace = 'masteriyo/pro/v1';

	/**
	 * Route base.
	 *
	 * @var string
	 */
	protected $rest_base = 'mailchimp_integration';

	/** Object type.
	 *
	 * @var string
	 */
	protected $object_type = 'mailchimp-integration';


	/**
	 * If object is hierarchical.
	 *
	 * @var bool
	 */
	protected $hierarchical = true;

	/**
	 * Permission class.
	 *
	 * @since 2.7.1
	 *
	 * @var \Masteriyo\Helper\Permission;
	 */
	protected $permission = null;

	/**
	 * Constructor.
	 *
	 * @since 2.7.1
	 *
	 * @param \Masteriyo\Helper\Permission $permission
	 */
	public function __construct( Permission $permission = null ) {
		$this->permission = $permission;
	}

	/**
	 * Register routes.
	 *
	 * @since 2.7.1
	 *
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			$this->namespace,
			'/' . $this->rest_base,
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_mailchimp_data' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
					'args'                => array(
						'force' => array(
							'type'    => 'boolean',
							'default' => false,
						),
					),
				),
			)
		);
	}

	/**
	 * Check if a given request has access to read items.
	 *
	 * @since 2.7.1
	 *
	 * @param  WP_REST_Request $request Full details about the request.
	 * @return WP_Error|boolean
	 */
	public function get_items_permissions_check( $request ) {
		if ( is_null( $this->permission ) ) {
			return new \WP_Error(
				'masteriyo_null_permission',
				__( 'Sorry, the permission object for this resource is null.', 'learning-management-system' )
			);
		}

		return current_user_can( 'manage_options' ) || current_user_can( 'manage_masteriyo_settings' );
	}


	/**
	 * Get mailchimps all audience and groups.
	 *
	 * @since 2.7.1
	 *
	 * @param  \WP_REST_Request $request Full details about the request.
	 * @return lists of audience and groups/tags
	 */
	public function get_mailchimp_data( $request ) {
		if ( $request->get_param( 'force' ) ) {
			delete_transient( '_masteriyo_mailchimp_data' );
		}

		$mailchimp_data = get_transient( '_masteriyo_mailchimp_data' );
		if ( empty( $mailchimp_data ) ) {
			$setting = new Setting();
			$api_key = $setting->get( 'api_key' );

			if ( ! empty( $api_key ) ) {
				$mailchimp_addon = new MailchimpIntegrationAddon();
				$mailchimp_addon->init();
				$mailchimp_data = $mailchimp_addon->get_mailchimp_audience_and_groups();
				set_transient( '_masteriyo_mailchimp_data', $mailchimp_data, WEEK_IN_SECONDS );

			}
		}
		return $mailchimp_data;
	}

}
