<?php
/**
 * Masteriyo White Label setup.
 *
 * @package Masteriyo\Addons\WhiteLabel
 *
 * @since 2.2.9
 */
namespace Masteriyo\Addons\WhiteLabel;

use Masteriyo\Constants;
use Masteriyo\Addons\WhiteLabel\GlobalSetting;

defined( 'ABSPATH' ) || exit;

defined( 'ABSPATH' ) || exit;
/**
 * Main Masteriyo WhiteLabel class.
 *
 * @class Masteriyo\Addons\WhiteLabel
 */
class WhiteLabelAddon {
	/**
	 * White label global setting instance.
	 *
	 * @since 2.2.9
	 *
	 * @var \Masteriyo\Addons\WhiteLabel\GlobalSetting
	 */
	public $global_setting = null;
	/**
	 * Initialize module.
	 *
	 * @since 2.2.9
	 */
	public function init() {
		$this->global_setting = new GlobalSetting();
		$this->global_setting->init();

		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 *
	 * @since 2.2.9
	 */
	public function init_hooks() {
		add_filter( 'masteriyo_admin_menu_title', array( $this, 'change_admin_menu_title' ) );
		add_filter( 'masteriyo_active_admin_menu_icon', array( $this, 'change_active_admin_menu_icon' ) );
		add_filter( 'masteriyo_inactive_admin_menu_icon', array( $this, 'change_inactive_admin_menu_icon' ) );
		add_action( 'after_setup_theme', array( $this, 'add_image_sizes' ) );
		add_filter( 'masteriyo_localized_admin_scripts', array( $this, 'localize_logo' ) );
		add_filter( 'masteriyo_enqueue_styles', array( $this, 'enqueue_styles' ) );
		add_filter( 'masteriyo_user_roles', array( $this, 'modify_user_roles' ) );
		add_filter( 'masteriyo_localized_admin_scripts', array( $this, 'add_backend_script_data' ), 10, 1 );
		add_filter( 'display_post_states', array( $this, 'masteriyo_add_whitelist_post_state' ), 10, 2 );
		add_filter( 'masteriyo_invoice_logo_url', array( $this, 'get_white_label_logo_url' ) );
	}

	/**
	 *  Add a post display state for special masteriyo pages in the page list table.
	 *
	 * @since 2.8.0
	 *
	 * @param array   $post_states An array of post display states.
	 * @param WP_Post $post        The current post object.
	 */
	public function masteriyo_add_whitelist_post_state( $post_states, $post ) {
		$white_label = $this->global_setting->get( 'title' );

		if ( empty( $white_label ) ) {
			return $post_states;
		}

		$new_post_states = array();

		foreach ( $post_states as $key => $post_state ) {
			$new_post_states[ $key ] = str_replace( 'Masteriyo', $white_label, $post_state );
		}

		return $new_post_states;
	}


	/**
	 * Getting title from the setting for white label.
	 *
	 * @since 2.8.0
	 * @param array $localized_scripts Localized admin scripts.
	 * @return array
	 */
	public function add_backend_script_data( $localized_admin_scripts ) {
			$localized_admin_scripts['backend']['data']['whiteLabelTitle'] = $this->global_setting->get( 'title' );
			return $localized_admin_scripts;
	}

	/**
	 * Change admin menu title.
	 *
	 * @since 2.2.9
	 *
	 * @param string $title Admin menu title.
	 * @return string
	 */
	public function change_admin_menu_title( $title ) {
		$setting_title = $this->global_setting->get( 'title' );

		if ( empty( $setting_title ) ) {
			$title = __( 'Masteriyo', 'learning-management-system' );
		} else {
			$title = $setting_title;
		}

		return $title;
	}
	/**
	 * Change active admin menu icon.
	 *
	 * @since 2.2.9
	 *
	 * @param string $icon Active admin menu icon.
	 * @return string
	 */
	public function change_active_admin_menu_icon( $icon ) {
		$active_icon     = $this->global_setting->get( 'active_icon' );
		$active_icon_url = wp_get_attachment_image_src( $active_icon, 'masteriyo_admin_menu' );
		$active_icon_url = ( false === $active_icon_url ) ? $icon : $active_icon_url[0];

		return $active_icon_url;
	}
	/**
	 * Change inactive admin menu icon.
	 *
	 * @since 2.2.9
	 *
	 * @param string $icon Inactive admin menu icon.
	 * @return string
	 */
	public function change_inactive_admin_menu_icon( $icon ) {
		$inactive_icon     = $this->global_setting->get( 'inactive_icon' );
		$inactive_icon_url = wp_get_attachment_image_src( $inactive_icon, 'masteriyo_admin_menu' );
		$inactive_icon_url = ( false === $inactive_icon_url ) ? $icon : $inactive_icon_url[0];

		return $inactive_icon_url;
	}

	/**
	 * Add image sizes.
	 *
	 * @since 2.2.9
	 */
	public function add_image_sizes() {
		add_image_size( 'masteriyo_admin_menu', 40, 40, true );
		add_image_size( 'masteriyo_logo', 127, 143, true );
	}

	/**
	 * Localize masteriyo logo.
	 *
	 * @since 2.2.9
	 * @since 2.9.1 Used new method to get logo url.
	 *
	 * @param array $localized_admin_scripts
	 * @return array
	 */
	public function localize_logo( $localized_admin_scripts ) {
		$logo_url = $this->get_white_label_logo_url( 'full' );

		$localized_admin_scripts['backend']['data']['logo'] = $logo_url;

		return $localized_admin_scripts;
	}

	/**
	 * Get white label logo url.
	 *
	 * @since 2.9.1
	 */
	public function get_white_label_logo_url( $size = 'masteriyo_logo' ) {
		$logo = wp_get_attachment_image_src( $this->global_setting->get( 'logo' ), $size );

		if ( false === $logo ) {
			$logo_url = plugins_url( 'assets/img/logo.png', MASTERIYO_PLUGIN_FILE );
		} else {
			$logo_url = $logo[0];
		}

		return $logo_url;
	}

	/**
	 * Add styles for enqueue.
	 *
	 * @since 2.7.0
	 *
	 * @param array $styles
	 *
	 * @return array
	 */
	public function enqueue_styles( $styles ) {
		$styles['masteriyo-white-label-admin'] = array(
			'src'      => plugins_url( 'addons/white-label/css/admin.css', MASTERIYO_PLUGIN_FILE ),
			'has_rtl'  => false,
			'context'  => 'admin',
			'callback' => array( $this, 'is_load_admin_css' ),
		);

		return $styles;
	}

	/**
	 * Check whether the admin css should be loaded for the current page.
	 *
	 * @since 2.7.0
	 *
	 * @return boolean
	 */
	public function is_load_admin_css() {
		return (
			! empty( $this->global_setting->get( 'active_icon' ) ) ||
			! empty( $this->global_setting->get( 'inactive_icon' ) )
		);
	}

	/**
	 * Modify role names according to the setting.
	 *
	 * @since 2.7.1
	 *
	 * @param array $roles
	 *
	 * @return array
	 */
	public function modify_user_roles( $roles ) {
		$student_role_name    = trim( $this->global_setting->get( 'student_role_name' ) );
		$instructor_role_name = trim( $this->global_setting->get( 'instructor_role_name' ) );

		if ( empty( $student_role_name ) ) {
			$roles['masteriyo_student']['display_name'] = esc_html__( 'Masteriyo Student', 'learning-management-system' );
		} else {
			$roles['masteriyo_student']['display_name'] = $student_role_name;
		}

		if ( empty( $instructor_role_name ) ) {
			$roles['masteriyo_instructor']['display_name'] = esc_html__( 'Masteriyo Instructor', 'learning-management-system' );
		} else {
			$roles['masteriyo_instructor']['display_name'] = $instructor_role_name;
		}

		return $roles;
	}
}
