<?php
/**
 * Two Factor Authentication OTP email class.
 *
 * @package Masteriyo\Emails
 *
 * @since 2.16.0
 */

namespace Masteriyo\Emails;

use Masteriyo\Abstracts\Email;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

/**
 * Two Factor Authentication OTP email class.
 *
 * @since 2.16.0
 *
 * @package Masteriyo\Emails
 */
class TwoFactorAuthenticationOTPEmail extends Email {
	/**
	 * Email method ID.
	 *
	 * @since 2.16.0
	 *
	 * @var String
	 */
	protected $id = 'two-factor-authentication-otp';

	/**
	 * HTML template path.
	 *
	 * @since 2.16.0
	 *
	 * @var string
	 */
	protected $html_template = 'emails/two-factor-authentication-otp.php';

	/**
	 * Send this email.
	 *
	 * @since 2.16.0
	 *
	 * @param \Masteriyo\Models\User $user The user user object.
	 */
	public function trigger( $user ) {
		$user = masteriyo_get_user( $user );

		// Bail early if user doesn't exist.
		if ( is_wp_error( $user ) || is_null( $user ) ) {
			return;
		}

		if ( empty( $user->get_email() ) ) {
			return;
		}

		$user_email           = $user->get_email();
		$to_addresses_setting = masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.to_address' );
		$to_address           = array();

		if ( ! empty( $to_addresses_setting ) ) {
			$to_addresses_setting = str_replace( '{user_email}', $user_email, $to_addresses_setting );
			$to_address           = explode( ',', $to_addresses_setting );
		}

		$this->set_recipients( ! empty( $to_address ) ? $to_address : $user_email );
		$this->set( 'email_heading', $this->get_heading() );
		$this->set( 'user', $user );

		$this->send(
			$this->get_recipients(),
			$this->get_subject(),
			$this->get_content(),
			$this->get_headers(),
			$this->get_attachments()
		);
	}

	/**
	 * Return true if it is enabled.
	 *
	 * @since 2.16.0
	 *
	 * @return bool
	 */
	public function is_enabled() {
		return masteriyo_string_to_bool( masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.enable' ) );
	}

	/**
	 * Return subject.
	 *
	 * @since 2.16.0
	 *
	 * @return string
	 */
	public function get_subject() {
		/**
		 * Filter 2FA OTP email subject.
		 *
		 * @since 2.16.0
		 *
		 * @param string $subject.
		 */
		$subject = apply_filters( $this->get_full_id(), masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.subject' ) );
		$subject = is_string( $subject ) ? trim( $subject ) : '';
		$subject = empty( $subject ) ? masteriyo_get_default_email_contents()['everyone']['two_factor_authentication_otp']['subject'] : $subject;

		return $this->format_string( $subject );
	}

	/**
	 * Return heading.
	 *
	 * @since 2.16.0
	 *
	 * @return string
	 */
	public function get_heading() {
		/**
		 * Filter 2FA OTP email heading.
		 *
		 * @since 2.16.0
		 *
		 * @param string $heading.
		 */
		$heading = apply_filters( $this->get_full_id(), masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.heading' ) );

		return $this->format_string( $heading );
	}

	/**
	 * Return additional content.
	 *
	 * @since 2.16.0
	 *
	 * @return string
	 */
	public function get_additional_content() {

		/**
		 * Filter 2FA OTP email additional content.
		 *
		 * @since 2.16.0
		 *
		 * @param string $additional_content.
		 */
		$additional_content = apply_filters( $this->get_full_id(), masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.additional_content' ) );

		return $this->format_string( $additional_content );
	}

	/**
	 * Get email content.
	 *
	 * @since 2.16.0
	 *
	 * @return string
	 */
	public function get_content() {
		$content = masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.content' );
		$content = $this->format_string( $content );

		$this->set( 'content', trim( $content ) );

		return parent::get_content();
	}

	/**
	 * Get placeholders.
	 *
	 * @since 2.16.0
	 *
	 * @return array
	 */
	public function get_placeholders() {
		$placeholders = parent::get_placeholders();

		/** @var \Masteriyo\Models\User $user */
		$user = $this->get( 'user' );

		if ( $user ) {
			$placeholders['{user_email}']          = $user->get_email();
			$placeholders['{username}']            = $user->get_username();
			$placeholders['{first_name}']          = $user->get_first_name();
			$placeholders['{last_name}']           = $user->get_last_name();
			$placeholders['{otp_code}']            = masteriyo_generate_otp_code( $user->get_id() );
			$placeholders['{otp_expiration_time}'] = masteriyo_get_login_otp_expiration_time( true );
		}

		return $placeholders;
	}

	/**
	 * Get the reply_to_name.
	 *
	 * @since 2.16.0
	 *
	 * @return string
	 */
	public function get_reply_to_name() {
		/**
		 * Filter student registration email reply_to_name to admin.
		 *
		 * @since 2.16.0
		 *
		 * @param string $reply_to_name.
		 */
		$reply_to_name = apply_filters( $this->get_full_id() . 'reply_to_name', masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.reply_to_name' ) );
		$reply_to_name = is_string( $reply_to_name ) ? trim( $reply_to_name ) : '';

		return ! empty( $reply_to_name ) ? wp_specialchars_decode( esc_html( $reply_to_name ), ENT_QUOTES ) : parent::get_reply_to_name();
	}

	/**
	 * Get the reply_to_address.
	 *
	 * @since 2.16.0
	 *
	 * @return string
	 */
	public function get_reply_to_address( $reply_to_address = '' ) {
		/**
		 * Filter student registration email reply_to_address to admin.
		 *
		 * @since 2.16.0
		 *
		 * @param string $reply_to_address.
		 */
		$reply_to_address = apply_filters( $this->get_full_id() . 'reply_to_address', masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.reply_to_address' ) );
		$reply_to_address = is_string( $reply_to_address ) ? trim( $reply_to_address ) : '';

		return ! empty( $reply_to_address ) ? sanitize_email( $reply_to_address ) : parent::get_reply_to_address();
	}

	/**
	 * Get the from_name.
	 *
	 * @since 2.16.0
	 *
	 * @return string
	 */
	public function get_from_name() {
		/**
		 * Filter student registration email from_name to admin.
		 *
		 * @since 2.16.0
		 *
		 * @param string $from_name.
		 */
		$from_name = apply_filters( $this->get_full_id() . '_from_name', masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.from_name' ) );
		$from_name = is_string( $from_name ) ? trim( $from_name ) : '';

		return ! empty( $from_name ) ? wp_specialchars_decode( esc_html( $from_name ), ENT_QUOTES ) : parent::get_from_name();
	}

	/**
	 * Get the from_address.
	 *
	 * @since 2.16.0
	 *
	 * @return string
	 */
	public function get_from_address( $from_address = '' ) {
		/**
		 * Filter student registration email from_address to admin.
		 *
		 * @since 2.16.0
		 *
		 * @param string $from_address.
		 */
		$from_address = apply_filters( $this->get_full_id() . '_from_address', masteriyo_get_setting( 'emails.everyone.two_factor_authentication_otp.from_address' ) );
		$from_address = is_string( $from_address ) ? trim( $from_address ) : '';

		return ! empty( $from_address ) ? sanitize_email( $from_address ) : parent::get_from_address();
	}
}
