<?php
/**
 * License store
 *
 * @since 2.5.0
 * @package Masteriyo\Pro
 */

namespace Masteriyo\Pro;

class License {

	/**
	 * Constructor.
	 *
	 * @since 2.5.0
	 */
	public function __construct() {
		$this->populate_activation_data_with_variant();
	}

	/**
	 * Populate license activation data with variant information.
	 *
	 * @since 2.5.0
	 *
	 * @return void
	 */
	public function populate_activation_data_with_variant() {
		$activation_data = $this->get_activation_data();

		if ( empty( $activation_data ) ) {
			return;
		}

		$variant_id = masteriyo_array_get( $activation_data, 'meta.variant_id', 0 );

		// Don't fetch if there's already a variant.
		if ( ! empty( $variant_id ) ) {
			return;
		}

		$order_id = masteriyo_array_get( $activation_data, 'meta.order_id', 0 );

		if ( empty( $order_id ) ) {
			return;
		}

		$data = masteriyo_pro_deactivate_license( $this->get_license_key(), $this->get_instance_id() );

		if ( is_wp_error( $data ) ) {
			return;
		}

		$data = masteriyo_pro_activate_license( $this->get_license_key(), masteriyo_pro_get_instance_name() );

		if ( is_wp_error( $data ) ) {
			return;
		}

		$this->set_activation_data( $data );
	}

	/*
	|--------------------------------------------------------------------------
	| Conditional
	|--------------------------------------------------------------------------
	*/

	/**
	 * Return true if the license is active.
	 *
	 * @since 2.5.0
	 *
	 * @return boolean
	 */
	public function is_license_active() {
		return 'active' === masteriyo_array_get( $this->get_activation_data(), 'license_key.status', 'inactive' );
	}

	/*
	|--------------------------------------------------------------------------
	| Getters
	|--------------------------------------------------------------------------
	*/

	/**
	 * Return license key.
	 *
	 * @since 2.5.0
	 *
	 * @return string
	 */
	public function get_license_key() {
		return get_option( 'masteriyo_pro_license_key', '' );
	}

	/**
	 * Return license status.
	 *
	 * @since 2.5.0
	 *
	 * @return string
	 */
	public function get_license_status() {
		$data = $this->get_activation_data();

		return masteriyo_array_get( $data, 'license_key.status', 'inactive' );
	}

	/**
	 * Return license activation data.
	 *
	 * @since 2.5.0
	 *
	 * @return data
	 */
	public function get_activation_data() {
		return get_option( 'masteriyo_pro_license_activation_data', array() );
	}

	/**
	 * Return instance id.
	 *
	 * @since 2.5.0
	 *
	 * @return void
	 */
	public function get_instance_id() {
		return masteriyo_array_get( $this->get_activation_data(), 'instance.id', null );
	}

	/**
	 * Return variant name.
	 *
	 * @since 2.5.0
	 *
	 * @return string
	 */
	public function get_variant_name() {
		return masteriyo_pro_get_variant_name_from_id( $this->get_variant_id() );
	}

	/*
	|--------------------------------------------------------------------------
	| Setters
	|--------------------------------------------------------------------------
	*/

	/**
	 * Set license key.
	 *
	 * @since 2.5.0
	 *
	 * @param string $license_key License key.
	 */
	public function set_license_key( $license_key ) {
		update_option( 'masteriyo_pro_license_key', $license_key );
	}

	/**
	 * Set license activation data.
	 *
	 * @since 2.0.1
	 *
	 * @param array $data License activation data.
	 */
	public function set_activation_data( $data ) {
		update_option( 'masteriyo_pro_license_activation_data', $data );
	}

	/**
	 * Return variant ID from activation data.
	 *
	 * @since 2.5.0
	 *
	 * @return array
	 */
	public function get_variant_id() {
		return masteriyo_array_get( $this->get_activation_data(), 'meta.variant_id', 0 );
	}

	/**
	 * Fetch variant information form order.
	 *
	 * @since 2.5.0
	 *
	 * @param int $order_id
	 * @return array
	 */
	public function activate_license() {
		$instance_name = masteriyo_pro_get_instance_name();
		$license_key   = $this->get_license_key();

		return masteriyo_pro_activate_license( $license_key, $instance_name );
	}

}
