<?php
/**
 * UserRegistrationFileUpload Functions.
 *
 * General core functions available on both the front-end and admin.
 *
 * @author   WPEverest
 * @category Core
 * @package  UserRegistrationFileUpload/Functions
 * @version  1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * File path.
 *
 * @param string $path Path to file.
 *
 * @return string
 */
function urfu_form_field_file( $path ) {

	return URFU_ABSPATH . 'includes/form/class-ur-file.php';
}

/**
 * Push file upload filed to registerd form fields.
 *
 * @param array $fields Fields list.
 */
function urfu_registered_form_fields( $fields ) {

	$field = 'file';

	if ( ! isset( $fields[ $field ] ) ) {

		array_push( $fields, $field );
	}

	return $fields;
}

/**
 * File upload settings in settings page.
 *
 * @return mixed
 */
function urfu_file_upload_settings() {
	$valid_file_type = get_option( 'user_registration_file_upload_setting_valid_file_type', urfu_get_valid_file_type() );

	$new_valid_types = array();

	foreach ( $valid_file_type as $key => $file_type ) {
		if ( isset( $file_type_array[ $file_type ] ) ) {
			$new_valid_types[ $key ] = $file_type_array[ $file_type ];
		}
	}

	if ( ! empty( $new_valid_types ) ) {
		update_option( 'user_registration_file_upload_setting_valid_file_type', $new_valid_types );
	}

	$max_upload_size_ini = wp_max_upload_size() / 1024;

	return apply_filters(
		'user_registration_file_upload_settings',
		array(
			'title'    => __( 'Upload Settings', 'user-registration-file-upload' ),
			'sections' => array(
				'user_registration_file_upload_settings' => array(
					'title'    => __( 'File Upload Settings', 'user-registration-file-upload' ),
					'type'     => 'card',
					'desc'     => '',
					'settings' => array(
						array(
							'title'    => __( 'Valid File Types', 'user-registration-file-upload' ),
							'desc'     => __( 'Choose valid file types allowed for uploads', 'user-registration-file-upload' ),
							'id'       => 'user_registration_file_upload_setting_valid_file_type',
							'default'  => array( 'pdf' ),
							'type'     => 'multiselect',
							'class'    => 'ur-enhanced-select',
							'css'      => 'min-width: 350px;',
							'desc_tip' => true,
							'options'  => urfu_get_valid_file_type(),
						),
						array(
							'title'    => __( 'Max File Size Allowed', 'user-registration-file-upload' ),
							'desc'     => __( 'Maximum file size allowed for upload. Enter file size in Kb. Leave the field empty for upload file size upto ' . $max_upload_size_ini . ' KB set in ini configuration.', 'user-registration-file-upload' ),
							'desc_tip' => true,
							'id'       => 'user_registration_file_upload_setting_max_file_size',
							'default'  => '1024',
							'type'     => 'number',
							'css'      => 'min-width: 350px;',
							'autoload' => false,
						),
					),
				),
			),
		)
	);
}

/**
 * Define Valid File types.
 *
 * @return mixed
 */
function urfu_get_valid_file_type() {

	return apply_filters(
		'user_registration_file_upload_valid_file_type',
		array(
			'application/pdf'                                                           => esc_html( 'pdf' ),
			'application/msword'                                                        => esc_html( 'doc' ),
			'application/vnd.openxmlformats-officedocument.wordprocessingml.document'   => esc_html( 'docx' ),
			'image/jpeg'                                                                => esc_html( 'jpg' ),
			'image/jpeg'                                                                => esc_html( 'jpeg' ),
			'image/gif'                                                                 => esc_html( 'gif' ),
			'image/png'                                                                 => esc_html( 'png' ),
			'audio/mpeg, audio/mpeg3, audio/x-mpeg-3, video/mpeg, video/x-mpeg, .mp3'  => esc_html( 'mp3' ),
			'video/mp4'                                                                 => esc_html( 'mp4' ),
			'text/csv,application_csv'                                                  => esc_html( 'csv' ),
			'text/plain'                                                                => esc_html( 'plain' ),
			'application/vnd.ms-powerpoint'                                             => esc_html( 'ppt' ),
			'application/vnd.openxmlformats-officedocument.presentationml.presentation' => esc_html( 'pptx' ),
			'application/vnd.ms-excel'                                                  => esc_html( 'xls' ),
			'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'         => esc_html( 'xlsx' ),
		)
	);
}

function user_registration_file_upload_check_file_types( $ext ) {
	$supportedFileTypes = array(
		'pdf',
		'doc',
		'xls',
		'ppt',
		'mp3',
		'mp4',
		'zip',
	);
	$newMsFileTypes     = array( 'docx', 'xlsx', 'pptx' );
	$imageFileTypes     = array( 'jpg', 'jpeg', 'png', 'gif' );

	$fileIcon = null;

	if ( in_array( $ext, $supportedFileTypes ) ) {
		$fileIcon = $ext;
	} elseif ( in_array( $ext, $newMsFileTypes ) ) {
		$fileIcon = substr( $ext, 0, - 1 );
	} elseif ( ! in_array( $ext, $imageFileTypes ) ) {
		$fileIcon = 'default';
	}

	return $fileIcon;
}

/**
 * Enqueue scripts.
 */
function urfu_register_upload_script() {
	$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';

	wp_register_script(
		'user-registration-file-upload-admin',
		URFU()->plugin_url() . '/assets/js/admin/user-registration-file-upload-admin' . $suffix . '.js',
		array(
			'jquery',
		),
		URFU_VERSION
	);

	wp_register_style( 'user-registration-file-upload-admin-style', URFU()->plugin_url() . '/assets/css/user-registration-file-upload-admin-style.css', array(), URFU_VERSION );
}

/**
 * Get all user registration forms.
 */
function urfu_get_all_user_registration_form() {

	$args = array(
		'post_type'   => 'user_registration',
		'status'      => 'publish',
		'numberposts' => - 1
	);

	$posts_array = get_posts( $args );

	foreach ( $posts_array as $post ) {
		$all_forms[ $post->ID ] = $post->post_title;
	}

	return $all_forms;
}

/**
 * Verify if uploaded file type is accepted by WordPress.
 *
 * @param array $file Uploaded file details.
 *
 * @return mixed
 */
function urfu_upload_file_type( $file ) {

	if ( isset( $_POST['urfu_upload'] ) && ur_string_to_bool( $_POST['urfu_upload'] ) ) {

		if ( isset( $_POST['urfu-meta-name'] ) && ! empty( $_POST['urfu-meta-name'] ) ) {
			$meta_name = sanitize_text_field( $_POST['urfu-meta-name'] );
			/*let's get the field details so we can see if we have any file restrictions */
			$all_fields = apply_filters(
				'urfu_form_fields',
				get_option( 'urfu_manage_fields' ),
				array(
					'context'          => 'upload_helper',
					'upload_meta_name' => $meta_name,
				)
			);

			if ( ! empty( $all_fields ) ) {

				foreach ( $all_fields as $field ) {

					if ( $field['meta-name'] == $meta_name ) {
						$allowed_upload_extensions = '';

						if ( 'Upload' == $field['field'] && ! empty( $field['allowed-upload-extensions'] ) ) {
							$allowed_upload_extensions = $field['allowed-upload-extensions'];
						}

						if ( 'Avatar' == $field['field'] && ! empty( $field['allowed-image-extensions'] ) ) {

							if ( trim( $field['allowed-image-extensions'] ) == '.*' ) {
								$allowed_upload_extensions = '.jpg,.jpeg,.gif,.png';
							} else {
								$allowed_upload_extensions = $field['allowed-image-extensions'];
							}
						}

						$ext = strtolower( substr( strrchr( $file['name'], '.' ), 1 ) );

						if ( ! empty( $allowed_upload_extensions ) && '.*' != $allowed_upload_extensions ) {
							$allowed = str_replace( '.', '', array_map( 'trim', explode( ',', strtolower( $allowed_upload_extensions ) ) ) );
							// first check if the user uploaded the right type
							if ( ! in_array( $ext, (array) $allowed ) ) {
								$file['error'] = __( 'Sorry, you cannot upload this file type for this field.', 'user-registration-file-upload' );

								return $file;
							}
						}
						// check if the type is allowed at all by WordPress.
						foreach ( get_allowed_mime_types() as $key => $value ) {

							if ( strpos( $key, $ext ) !== false || $key == $ext ) {
								return $file;
							}
						}
						$file['error'] = __( 'Sorry, you cannot upload this file type for this field.', 'user-registration-file-upload' );
					}
				}
			}
		}

		if ( empty( $_POST['urfu-meta-name'] ) ) {
			$file['error'] = __( 'An error occurred, please try again later.', 'user-registration-file-upload' );
		}
	}

	return $file;
}

/**
 * Check if file upload addon is compatible.
 *
 * @return string
 */
function urfu_is_compatible() {

	$ur_pro_plugins_path = WP_PLUGIN_DIR . URFU_DS . 'user-registration-pro' . URFU_DS . 'user-registration.php';

	if ( ! file_exists( $ur_pro_plugins_path ) ) {
		return;
	}

	$ur_pro_plugin_file_path = 'user-registration-pro/user-registration.php';

	include_once ABSPATH . 'wp-admin/includes/plugin.php';

	if ( ! is_plugin_active( $ur_pro_plugin_file_path ) ) {
		return;
	}

	if ( function_exists( 'UR' ) ) {
		$user_registration_version = UR()->version;
	} else {
		$user_registration_version = get_option( 'user_registration_version' );
	}

	if ( version_compare( $user_registration_version, '4.0.0', '<' ) ) {
		return;
	}

	return 'YES';
}

/**
 * Action to print compatibility notices.
 */
function urfu_check_plugin_compatibility() {

	add_action( 'admin_notices', 'user_registration_file_upload_admin_notice', 10 );
}

/**
 * Print compatibility notices.
 */
function user_registration_file_upload_admin_notice() {

	$class = 'error notice is-dismissible';

	$message = urfu_is_compatible();

	if ( 'YES' !== $message ) {

		printf( '<div class="' . esc_attr( $class ) . '"><p>' . sprintf( esc_html__( 'User Registration File Upload  requires %s version 4.0.0 or greater to work', 'user-registration-file-upload' ), '<a href="https://wpuserregistration.com/" target="_blank">' . esc_html__( 'User Registration Pro', 'user-registration-file-upload' ) . '</a>' ) . '</p></div>' );

	}
}

/**
 * Deprecate plugin missing notice.
 *
 * @return void
 * @deprecated 1.1.2
 *
 */
function urfu_admin_notices() {
	ur_deprecated_function( 'urfu_admin_notices', '1.1.2', 'user_registration_file_upload_admin_notice' );
}
