<?php
/**
 * User registration Woocommerce blocks.
 *
 * @since 3.1.5
 * @package user-registration-woocommerce
 */

defined( 'ABSPATH' ) || exit;
/**
 * User registration woocommerce blocks class.
 */
class URWC_Blocks {

	/**
	 * Default WC REST Request array.
	 * Stores WC REST Request based on user registration form data.
	 *
	 * @var [array]
	 */
	public $ur_request = array();
	/**
	 * Constructor.
	 */
	public function __construct() {
		include_once URWC_ABSPATH . 'includes/blocks/block-types/class-urwc-block-abstract.php';
		include_once URWC_ABSPATH . 'includes/blocks/block-types/class-urwc-block-checkout-sync-fields.php';
		$this->init_hooks();
		add_action( 'save_post', array( $this, 'get_block_attributes_after_save' ) );
		add_action( 'urwc_sync_check_fields_with_blocks_data', array( $this, 'sync_check_fields_with_blocks_data' ), 10, 2 );
	}

	/**
	 * Sync the check fields with block data.
	 *
	 * @param int   $form_id  The form id.
	 * @param array $checkout_fields Check fields list.
	 */
	public function sync_check_fields_with_blocks_data( $form_id, $checkout_fields ) {

		if ( ! ur_string_to_bool( get_option( 'user_registration_woocommrece_settings_sync_checkout', false ) ) ) {
			return;
		}

		$checkout_page_id = wc_get_page_id( 'checkout' );

		if ( empty( $checkout_page_id ) ) {
			return;
		}

		$post_content = get_post_field( 'post_content', $checkout_page_id );

		if ( strpos( $post_content, 'wp:urwc/checkout-sync-field' ) !== false ) {
			$selected_fields_json = json_encode( $checkout_fields );

			$post_content = preg_replace_callback(
				'/<!-- wp:urwc\/checkout-sync-field (.*?) -->/is',
				function ( $matches ) use ( $form_id, $selected_fields_json ) {
					$block_attributes = $matches[1];

					$block_attributes = preg_replace( '/"formId":"\d+"/', '"formId":"' . $form_id . '"', $block_attributes );

					$block_attributes = preg_replace( '/"selectedFields":\[[^\]]*\]/', '"selectedFields":' . $selected_fields_json, $block_attributes );

					return '<!-- wp:urwc/checkout-sync-field ' . $block_attributes . ' -->';
				},
				$post_content
			);

			if ( '' != $post_content ) {
				wp_update_post(
					array(
						'ID'           => $checkout_page_id,
						'post_content' => $post_content,
					)
				);
			}
		}
	}

	/**
	 * Get block attributes after save.
	 *
	 * @since 0
	 *
	 * @param  [type] $post_id The checkout page id.
	 */
	public function get_block_attributes_after_save( $post_id ) {

		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		if ( 'page' !== get_post_type( $post_id ) ) {
			return;
		}

		$post_content = get_post_field( 'post_content', $post_id );

		preg_match_all( '/<!-- wp:urwc\/checkout-sync-field ({.*?) -->/s', $post_content, $matches );

		$results = array();

		foreach ( $matches[1] as $match ) {
			$attributes = json_decode( $match, true );
			if ( $attributes ) {
				$results[] = $attributes;
			}
		}

		if ( isset( $results['0'] ) && ! empty( $results['0'] ) ) {
			$sync_data = $results['0'];
			if ( isset( $sync_data['formId'] ) && ! empty( $sync_data['formId'] ) ) {
				$form_id = absint( $sync_data['formId'] );
				update_option( 'user_registration_woocommerce_settings_form', $form_id );
				$checkout_form_fields                       = get_option( 'user_registration_woocommerce_checkout_fields', array() );
				$checkout_form_fields[ 'form-' . $form_id ] = isset( $sync_data['selectedFields'] ) ? $sync_data['selectedFields'] : array();
				update_option( 'user_registration_woocommerce_checkout_fields', $checkout_form_fields );

			}
		}
	}


	/**
	 * Init hooks.
	 *
	 * @since 3.1.5
	 */
	private function init_hooks() {
		add_filter( 'block_categories_all', array( $this, 'block_categories' ), PHP_INT_MAX, 2 );
		add_action( 'init', array( $this, 'register_block_types' ) );
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );
		add_action(
			'woocommerce_blocks_loaded',
			function () {
				require_once __DIR__ . '/class-urwc-extend-store-endpoint.php';
				require_once __DIR__ . '/class-urwc-blocks-integration.php';

				// Initialize our store endpoint extension when WC Blocks is loaded.
				URWC_Extend_Store_Endpoint::init();

				add_action(
					'woocommerce_blocks_checkout_block_registration',
					function ( $integration_registry ) {
						$integration_registry->register( new URWC_Blocks_Integration() );
					}
				);
			}
		);
		add_action( 'woocommerce_store_api_checkout_update_order_from_request', array( $this, 'urwc_get_wc_rest_request' ), 10, 2 );
		add_action( 'woocommerce_created_customer', array( $this, 'save_checkout_sync_fields' ), 10, 3 );
	}

	/**
	 * Save user information.
	 *
	 * @param int   $customer_id Customer ID.
	 * @param mixed $new_customer_data Customer Data.
	 * @param mixed $password_generated Password Generated.
	 */
	public function save_checkout_sync_fields( $customer_id, $new_customer_data, $password_generated ) {

		if ( empty( $this->ur_request ) ) {
			return;
		}

		$form_data = array();
		foreach ( $this->ur_request as $key => $form_value ) {
			$key               = str_replace( '[]', '', $key );
			$form_data[ $key ] = $form_value;
		}

		$form_id               = get_option( 'user_registration_woocommerce_settings_form', 0 );
		$sync_checkout_enabled = ur_string_to_bool( get_option( 'user_registration_woocommrece_settings_sync_checkout', false ) );

		if ( 0 < $form_id && $sync_checkout_enabled ) {

			$profile         = urwc_get_form_fields( $form_id );
			$valid_form_data = array();
			$hide_field_list = isset($form_data['urcl_hide_fields']) ? json_decode( stripslashes( $form_data['urcl_hide_fields'] ) ) : array(); // phpcs:ignore

			foreach ( $form_data as $post_key => $post_data ) {

				// Fetch Email and First/Last name from and replace with ur default field keys.
				if ( 'billing_email' === $post_key ) {
					$post_key = 'user_registration_user_email';
				} elseif ( 'billing_first_name' === $post_key || 'billing_last_name' === $post_key ) {
					$post_key = 'billing_first_name' === $post_key ? 'user_registration_first_name' : 'user_registration_last_name';
				}

				$pos = strpos( $post_key, 'user_registration_' );

				if ( false !== $pos && isset( $profile[ $post_key ]['field_key'] ) ) {
					$new_string = substr_replace( $post_key, '', $pos, strlen( 'user_registration_' ) );

					if ( ! empty( $new_string ) ) {
						if ( in_array( $new_string, $hide_field_list, true ) ) {
							continue;
						}
						$tmp_array       = ur_get_valid_form_data_format( $new_string, $post_key, $profile, $post_data );
						$valid_form_data = array_merge( $valid_form_data, $tmp_array );
					}
				}
			}

			$post_content = ur_get_post_content( $form_id );
			$files        = ur_get_form_data_by_key( $post_content, 'file' );

			foreach ( $files as $field_name => $value ) {
				if ( in_array( $field_name, $hide_field_list, true ) ) {
					continue;
				}
				$file_upload = isset( $form_data[ 'urfu_uploaded_file_' . $field_name ] ) && ! empty( $form_data[ 'urfu_uploaded_file_' . $field_name ] ) && is_array( $form_data[ 'urfu_uploaded_file_' . $field_name ] ) ? implode( ',', $form_data[ 'urfu_uploaded_file_' . $field_name ] ) : $form_data[ 'urfu_uploaded_file_' . $field_name ];

				$valid_form_data[ $field_name ]             = new stdClass();
				$valid_form_data[ $field_name ]->value      = $file_upload;
				$valid_form_data[ $field_name ]->field_name = $field_name;
			}
			if ( count( $valid_form_data ) > 0 ) {
				UR_Frontend_Form_Handler::ur_update_user_meta( $customer_id, $valid_form_data, $form_id );
				do_action( 'user_registration_after_user_meta_update', $valid_form_data, $form_id, $customer_id );
				do_action( 'user_registration_after_register_user_action', $valid_form_data, $form_id, $customer_id );
			}

			return;
		}
	}

	/**
	 * Retrive user registration sync field request from wc rest request.
	 *
	 * @param \WC_Order        $order ORDER DATA.
	 * @param \WP_REST_Request $request Request.
	 */
	public function urwc_get_wc_rest_request( \WC_Order $order, \WP_REST_Request $request ) {
		$ur_sync_field_request_data = isset( $request['extensions']['urwc-checkout-sync-field']['form-data'] ) ? $request['extensions']['urwc-checkout-sync-field']['form-data'] : array();
		if ( empty( $ur_sync_field_request_data ) ) {
			return;
		}

		$this->ur_request = $ur_sync_field_request_data;
	}

	/**
	 * Enqueue Block Editor Assets.
	 *
	 * @return void.
	 */
	public function enqueue_block_editor_assets() {
		global $pagenow;
		$enqueue_script = array( 'wp-blocks', 'wp-element', 'wp-i18n', 'wp-editor', 'wp-components', 'react', 'react-dom', 'selectWoo', 'jquery' );

		if ( wp_script_is( 'user-registration-advanced-fields-frontend', 'registered' ) ) {
			$enqueue_script[] = 'user-registration-advanced-fields-frontend';
		}

		wp_register_style(
			'urwc-blocks-editor',
			URWC()->plugin_url() . '/assets/css/user-registration.css',
			array( 'wp-edit-blocks', 'select2' ),
			URWC_VERSION
		);

		wp_enqueue_style( 'select2', UR()->plugin_url() . '/assets/css/select2/select2.css', array(), '4.0.6' );

		if ( defined( 'URFU_VERSION' ) ) {
			wp_register_style( 'user-registration-file-upload-style', URFU()->plugin_url() . '/assets/css/user-registration-file-upload-style.css', array(), URFU_VERSION );

			wp_enqueue_style( 'user-registration-file-upload-style' );
		}
			wp_register_script(
				'urwc-blocks-editor',
				URWC()->plugin_url() . '/chunks/blocks.js',
				$enqueue_script,
				URWC_VERSION
			);
			wp_localize_script(
				'urwc-blocks-editor',
				'_URWC_BLOCKS_',
				array(
					'logoUrl'                  => URWC()->plugin_url() . '/assets/images/logo.png',
					'urwcRestApiNonce'         => wp_create_nonce( 'wp_rest' ),
					'restURL'                  => rest_url(),
					'checkoutFormId'           => get_option( 'user_registration_woocommerce_settings_form', 0 ),
					'SyncCheckoutRegistration' => ur_string_to_bool( get_option( 'user_registration_woocommrece_settings_sync_checkout', false ) ),
					'globalSettingUrl'         => admin_url( 'admin.php?page=user-registration-settings&tab=woocommerce' ),
				)
			);
			wp_enqueue_style( 'urwc-blocks-editor' );
			wp_enqueue_script( 'urwc-blocks-editor' );
	}

	/**
	 * Add "User Registration" category to the blocks listing in post edit screen.
	 *
	 * @param array $block_categories All registered block categories.
	 * @return array
	 * @since 3.1.5
	 */
	public function block_categories( array $block_categories ) {
		return array_merge(
			array(
				array(
					'slug'  => 'user-registration-woocommerce',
					'title' => esc_html__( 'User Registration Woocommerce', 'user-registration-woocommerce' ),
				),
			),
			$block_categories
		);
	}

	/**
	 * Register block types.
	 *
	 * @return void
	 */
	public function register_block_types() {
		$block_types = $this->get_block_types();
		foreach ( $block_types as $block_type ) {
			new $block_type();
		}
	}

	/**
	 * Get block types.
	 *
	 * @return AbstractBlock[]
	 */
	private function get_block_types() {
		return apply_filters(
			'urwc_block_types',
			array(
			URWC_Block_Checkout_Sync_Fields::class, //phpcs:ignore;
			)
		);
	}
}
return new URWC_Blocks();
